<?php

namespace Tests\Feature;

use App\Models\ServiceRecord;
use App\Models\ServiceStatusChange;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class ServiceRecordTest extends TestCase
{
    use RefreshDatabase;

    protected $admin;
    protected $technician;
    protected $serviceRecord;
    protected $serviceManager;
    protected $serviceStaff;
    protected $customer;

    protected function setUp(): void
    {
        parent::setUp();

        // Admin kullanıcısı oluştur
        $this->admin = User::factory()->create([
            'role' => 'admin',
            'email' => 'admin_' . uniqid() . '@example.com'
        ]);

        // Servis yöneticisi oluştur
        $this->serviceManager = User::factory()->create([
            'role' => 'service_manager',
            'email' => 'manager_' . uniqid() . '@example.com'
        ]);

        // Servis personeli oluştur
        $this->serviceStaff = User::factory()->create([
            'role' => 'service_staff',
            'email' => 'staff_' . uniqid() . '@example.com'
        ]);

        // Teknik servis personeli oluştur
        $this->technician = User::factory()->create([
            'role' => 'technician',
            'email' => 'technician_' . uniqid() . '@example.com'
        ]);

        // Müşteri oluştur
        $this->customer = User::factory()->create([
            'role' => 'customer',
            'email' => 'customer_' . uniqid() . '@example.com'
        ]);

        // Servis kaydı oluştur
        $this->serviceRecord = ServiceRecord::factory()->create([
            'status' => ServiceRecord::STATUS_PENDING
        ]);
    }

    public function test_can_change_service_status()
    {
        $this->actingAs($this->admin);

        // Durum değişikliği yap
        $this->serviceRecord->changeStatus(
            ServiceRecord::STATUS_IN_PROGRESS,
            'İşleme alındı',
            $this->admin
        );

        // Durumun değiştiğini kontrol et
        $this->assertEquals(ServiceRecord::STATUS_IN_PROGRESS, $this->serviceRecord->status);

        // Durum değişikliği kaydının oluşturulduğunu kontrol et
        $this->assertDatabaseHas('service_status_changes', [
            'service_record_id' => $this->serviceRecord->id,
            'old_status' => ServiceRecord::STATUS_PENDING,
            'new_status' => ServiceRecord::STATUS_IN_PROGRESS,
            'description' => 'İşleme alındı',
            'changed_by_id' => $this->admin->id
        ]);
    }

    public function test_can_assign_technician()
    {
        $this->actingAs($this->admin);

        $serviceRecord = ServiceRecord::factory()->create([
            'status' => ServiceRecord::STATUS_PENDING,
            'technician_id' => null
        ]);

        $serviceRecord->assignTechnician(
            $this->technician,
            $this->admin,
            'Test assignment'
        );

        $this->assertDatabaseHas('service_records', [
            'id' => $serviceRecord->id,
            'technician_id' => $this->technician->id
        ]);
    }

    public function test_cannot_assign_non_technician()
    {
        $this->actingAs($this->admin);

        $serviceRecord = ServiceRecord::factory()->create([
            'status' => ServiceRecord::STATUS_PENDING,
            'technician_id' => null
        ]);

        $this->expectException(\InvalidArgumentException::class);
        $this->expectExceptionMessage('Only technicians can be assigned to service records.');

        $serviceRecord->assignTechnician(
            $this->customer,
            $this->admin,
            'Test assignment'
        );
    }

    public function test_can_get_status_text()
    {
        $this->assertEquals('Beklemede', $this->serviceRecord->statusText);

        $this->serviceRecord->status = ServiceRecord::STATUS_IN_PROGRESS;
        $this->assertEquals('İşlemde', $this->serviceRecord->statusText);
    }

    public function test_can_filter_by_status()
    {
        // Farklı durumlarda servis kayıtları oluştur
        ServiceRecord::factory()->create(['status' => ServiceRecord::STATUS_PENDING]);
        ServiceRecord::factory()->create(['status' => ServiceRecord::STATUS_IN_PROGRESS]);
        ServiceRecord::factory()->create(['status' => ServiceRecord::STATUS_COMPLETED]);

        // Bekleyen kayıtları filtrele
        $pendingRecords = ServiceRecord::pending()->get();
        $this->assertCount(2, $pendingRecords); // Başlangıçta oluşturulan + yeni oluşturulan

        // İşlemde olan kayıtları filtrele
        $inProgressRecords = ServiceRecord::inProgress()->get();
        $this->assertCount(1, $inProgressRecords);

        // Tamamlanan kayıtları filtrele
        $completedRecords = ServiceRecord::completed()->get();
        $this->assertCount(1, $completedRecords);
    }

    public function test_can_filter_by_technician_assignment()
    {
        // Önce mevcut servis kaydını temizle
        $this->serviceRecord->delete();

        // Teknik servis atanmış ve atanmamış kayıtlar oluştur
        $assignedRecord = ServiceRecord::factory()->create(['technician_id' => $this->technician->id]);
        $unassignedRecord = ServiceRecord::factory()->create(['technician_id' => null]);

        // Atanmış kayıtları filtrele
        $assignedRecords = ServiceRecord::assigned()->get();
        $this->assertCount(1, $assignedRecords);
        $this->assertEquals($assignedRecord->id, $assignedRecords->first()->id);

        // Atanmamış kayıtları filtrele
        $unassignedRecords = ServiceRecord::unassigned()->get();
        $this->assertCount(1, $unassignedRecords);
        $this->assertEquals($unassignedRecord->id, $unassignedRecords->first()->id);
    }

    public function test_can_filter_by_technician()
    {
        // Önce mevcut servis kaydını temizle
        $this->serviceRecord->delete();

        // Farklı teknisyenlere atanmış kayıtlar oluştur
        $technician2 = User::factory()->technician()->create();

        $record1 = ServiceRecord::factory()->create(['technician_id' => $this->technician->id]);
        $record2 = ServiceRecord::factory()->create(['technician_id' => $technician2->id]);

        // Belirli bir teknisyene atanmış kayıtları filtrele
        $technicianRecords = ServiceRecord::assignedTo($this->technician->id)->get();
        $this->assertCount(1, $technicianRecords);
        $this->assertEquals($record1->id, $technicianRecords->first()->id);
    }

    public function test_can_filter_by_date_range()
    {
        // Farklı tarihlerde kayıtlar oluştur
        ServiceRecord::factory()->create(['created_at' => now()->subDays(5)]);
        ServiceRecord::factory()->create(['created_at' => now()->subDays(3)]);
        ServiceRecord::factory()->create(['created_at' => now()->subDays(1)]);

        // Tarih aralığına göre filtrele
        $dateFrom = now()->subDays(4)->format('Y-m-d');
        $dateTo = now()->subDays(2)->format('Y-m-d');

        $filteredRecords = ServiceRecord::filterByDateRange($dateFrom, $dateTo)->get();
        $this->assertCount(1, $filteredRecords);
    }

    public function test_can_search_records()
    {
        // Farklı bilgilerle kayıtlar oluştur
        ServiceRecord::factory()->create([
            'customer_name' => 'Ahmet Yılmaz',
            'device_model' => 'iPhone 12'
        ]);
        ServiceRecord::factory()->create([
            'customer_name' => 'Mehmet Demir',
            'device_model' => 'Samsung S21'
        ]);

        // Arama yap
        $searchResults = ServiceRecord::search('Ahmet')->get();
        $this->assertCount(1, $searchResults);

        $searchResults = ServiceRecord::search('Samsung')->get();
        $this->assertCount(1, $searchResults);
    }

    public function test_can_get_status_badge_class()
    {
        $this->assertEquals('bg-yellow-100 text-yellow-800', $this->serviceRecord->statusBadgeClass);

        $this->serviceRecord->status = ServiceRecord::STATUS_IN_PROGRESS;
        $this->assertEquals('bg-blue-100 text-blue-800', $this->serviceRecord->statusBadgeClass);

        $this->serviceRecord->status = ServiceRecord::STATUS_COMPLETED;
        $this->assertEquals('bg-green-100 text-green-800', $this->serviceRecord->statusBadgeClass);
    }

    public function test_can_generate_unique_tracking_number()
    {
        $serviceRecord1 = ServiceRecord::factory()->create();
        $serviceRecord2 = ServiceRecord::factory()->create();

        $this->assertNotNull($serviceRecord1->tracking_number);
        $this->assertNotNull($serviceRecord2->tracking_number);
        $this->assertNotEquals($serviceRecord1->tracking_number, $serviceRecord2->tracking_number);
    }

    public function test_can_check_service_completion_status()
    {
        $serviceRecord = ServiceRecord::factory()->create([
            'status' => ServiceRecord::STATUS_COMPLETED
        ]);

        $this->assertTrue($serviceRecord->isCompleted());
        
        $serviceRecord->update(['status' => ServiceRecord::STATUS_PENDING]);
        $this->assertFalse($serviceRecord->isCompleted());
    }

    public function test_can_check_service_cancellation_status()
    {
        $serviceRecord = ServiceRecord::factory()->create([
            'status' => ServiceRecord::STATUS_CANCELLED
        ]);

        $this->assertTrue($serviceRecord->isCancelled());
        
        $serviceRecord->update(['status' => ServiceRecord::STATUS_PENDING]);
        $this->assertFalse($serviceRecord->isCancelled());
    }

    public function test_can_filter_by_multiple_criteria()
    {
        $technician = User::factory()->technician()->create();
        
        // Farklı durumlarda ve atamalarda test kayıtları oluştur
        $record1 = ServiceRecord::factory()->create([
            'status' => ServiceRecord::STATUS_PENDING,
            'technician_id' => $technician->id,
            'created_at' => now()->subDays(5)
        ]);
        
        $record2 = ServiceRecord::factory()->create([
            'status' => ServiceRecord::STATUS_IN_PROGRESS,
            'technician_id' => null,
            'created_at' => now()->subDays(2)
        ]);
        
        $record3 = ServiceRecord::factory()->create([
            'status' => ServiceRecord::STATUS_COMPLETED,
            'technician_id' => $technician->id,
            'created_at' => now()
        ]);

        // Çoklu filtreleme testi
        $filteredRecords = ServiceRecord::query()
            ->filterByStatus(ServiceRecord::STATUS_PENDING)
            ->filterByTechnician($technician->id)
            ->filterByDateRange(
                now()->subDays(7)->toDateString(),
                now()->subDays(3)->toDateString()
            )
            ->get();

        $this->assertCount(1, $filteredRecords);
        $this->assertEquals($record1->id, $filteredRecords->first()->id);
    }

    public function test_status_change_creates_history_record()
    {
        $this->actingAs($this->admin);

        $serviceRecord = ServiceRecord::factory()->create([
            'status' => ServiceRecord::STATUS_PENDING
        ]);

        $description = 'Test durum değişikliği';
        $serviceRecord->changeStatus(ServiceRecord::STATUS_IN_PROGRESS, $description, $this->admin);

        $this->assertDatabaseHas('service_status_changes', [
            'service_record_id' => $serviceRecord->id,
            'old_status' => ServiceRecord::STATUS_PENDING,
            'new_status' => ServiceRecord::STATUS_IN_PROGRESS,
            'description' => $description,
            'changed_by_id' => $this->admin->id
        ]);
    }

    public function test_cannot_assign_deleted_technician()
    {
        $this->actingAs($this->admin);

        $serviceRecord = ServiceRecord::factory()->create([
            'status' => ServiceRecord::STATUS_PENDING,
            'technician_id' => null
        ]);

        $deletedTechnician = User::factory()->create([
            'role' => 'technician',
            'email' => 'deleted@example.com'
        ]);
        $deletedTechnician->delete();

        $this->expectException(\InvalidArgumentException::class);
        $this->expectExceptionMessage('Cannot assign a deleted technician.');

        $serviceRecord->assignTechnician(
            $deletedTechnician,
            $this->admin,
            'Test assignment'
        );
    }
} 