<?php

namespace Tests\Feature;

use Tests\TestCase;
use Spatie\Permission\Models\Role;
use App\Models\User;
use App\Models\Technician;
use App\Models\Customer;
use App\Models\ServiceRecord;
use Illuminate\Support\Facades\Storage;
use PHPUnit\Framework\Attributes\Test;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Config;
use Spatie\Permission\PermissionRegistrar;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Str;

class ServiceRecordReportTest extends TestCase
{
    protected $admin;
    protected $technician;
    protected $customer;

    protected function setUp(): void
    {
        parent::setUp();

        // API guard'ı ayarla
        Config::set('auth.defaults.guard', 'api');

        // Cache'i devre dışı bırak
        Config::set('cache.default', 'array');
        Config::set('cache.stores.array', [
            'driver' => 'array'
        ]);

        // Spatie Permission cache'ini devre dışı bırak
        Config::set('permission.cache.expiration_time', 0);
        Config::set('permission.cache.store', null);
        Config::set('permission.enable_cache', false);

        // Cache'i temizle
        Cache::flush();

        // Spatie Permission cache'ini temizle
        app()[PermissionRegistrar::class]->forgetCachedPermissions();

        // Rolleri ve izinleri oluştur
        $this->createRolesAndPermissions();

        // Test verilerini oluştur
        $this->createTestData();
    }

    protected function createRolesAndPermissions()
    {
        // Rolleri oluştur
        $adminRole = Role::create(['name' => 'admin', 'guard_name' => 'api']);
        $technicianRole = Role::create(['name' => 'technician', 'guard_name' => 'api']);
        $customerRole = Role::create(['name' => 'customer', 'guard_name' => 'api']);
        $serviceManagerRole = Role::create(['name' => 'service_manager', 'guard_name' => 'api']);
        $serviceStaffRole = Role::create(['name' => 'service_staff', 'guard_name' => 'api']);
    }

    protected function createTestData()
    {
        // Kullanıcıları oluştur
        $this->admin = User::factory()->create([
            'name' => 'Admin User',
            'email' => 'admin_' . Str::random(10) . '@example.com',
            'password' => bcrypt('password')
        ]);
        $this->admin->assignRole('admin');

        $this->technician = User::factory()->create([
            'name' => 'Test Technician',
            'email' => 'technician_' . Str::random(10) . '@example.com',
            'password' => bcrypt('password')
        ]);
        $this->technician->assignRole('technician');

        $this->customer = User::factory()->create([
            'name' => 'Test Customer',
            'email' => 'customer_' . Str::random(10) . '@example.com',
            'password' => bcrypt('password')
        ]);
        $this->customer->assignRole('customer');

        // Teknisyen ve müşteri kayıtlarını oluştur
        $technician = Technician::factory()->create([
            'user_id' => $this->technician->id,
            'name' => $this->technician->name,
            'email' => $this->technician->email,
            'phone' => '5551234567',
            'specialization' => 'Test Specialization'
        ]);

        $customer = Customer::factory()->create([
            'user_id' => $this->customer->id,
            'name' => $this->customer->name,
            'email' => $this->customer->email,
            'phone' => '5559876543',
            'address' => 'Test Address'
        ]);

        // Servis kayıtlarını oluştur
        ServiceRecord::factory()->count(10)->create([
            'technician_id' => $technician->id,
            'customer_id' => $customer->id,
            'created_by' => $this->admin->id,
            'status' => 'completed',
            'completion_time' => 120,
            'scheduled_date' => now(),
            'completion_date' => now()->addDays(5)
        ]);

        ServiceRecord::factory()->count(5)->create([
            'technician_id' => $technician->id,
            'customer_id' => $customer->id,
            'created_by' => $this->admin->id,
            'status' => 'in_progress',
            'completion_time' => null,
            'scheduled_date' => now()->addDays(1)
        ]);

        ServiceRecord::factory()->count(3)->create([
            'technician_id' => $technician->id,
            'customer_id' => $customer->id,
            'created_by' => $this->admin->id,
            'status' => 'pending',
            'completion_time' => null,
            'scheduled_date' => now()->addDays(2)
        ]);
    }

    protected function tearDown(): void
    {
        parent::tearDown();
    }

    #[Test]
    public function test_can_get_status_distribution_statistics()
    {
        $response = $this->actingAs($this->admin, 'api')
            ->withoutExceptionHandling()
            ->getJson('/api/statistics/status-distribution');

        $response->assertStatus(200)
            ->assertJsonStructure([
                'data' => [
                    'completed',
                    'in_progress',
                    'pending',
                    'cancelled'
                ]
            ]);
    }

    #[Test]
    public function test_can_get_technician_performance_statistics()
    {
        $response = $this->actingAs($this->admin, 'api')
            ->withoutExceptionHandling()
            ->getJson('/api/statistics/technician-performance');

        $response->assertStatus(200)
            ->assertJsonStructure([
                'data' => [
                    '*' => [
                        'technician_name',
                        'completed_services',
                        'average_completion_time',
                        'customer_satisfaction'
                    ]
                ]
            ]);
    }

    #[Test]
    public function test_can_get_completion_time_statistics()
    {
        $response = $this->actingAs($this->admin, 'api')
            ->withoutExceptionHandling()
            ->getJson('/api/statistics/completion-time');

        $response->assertStatus(200)
            ->assertJsonStructure([
                'data' => [
                    'average_completion_time',
                    'min_completion_time',
                    'max_completion_time'
                ]
            ]);
    }

    #[Test]
    public function test_can_get_monthly_statistics()
    {
        $response = $this->actingAs($this->admin, 'api')
            ->withoutExceptionHandling()
            ->getJson('/api/statistics/monthly');

        $response->assertStatus(200)
            ->assertJsonStructure([
                'data' => [
                    '*' => [
                        'month',
                        'total_services',
                        'completed_services',
                        'revenue'
                    ]
                ]
            ]);
    }

    #[Test]
    public function test_can_get_customer_statistics()
    {
        $response = $this->actingAs($this->admin, 'api')
            ->withoutExceptionHandling()
            ->getJson('/api/statistics/customers');

        $response->assertStatus(200)
            ->assertJsonStructure([
                'data' => [
                    'total_customers',
                    'active_customers',
                    'new_customers_this_month',
                    'customer_satisfaction'
                ]
            ]);
    }

    #[Test]
    public function test_can_filter_statistics_by_date_range()
    {
        $response = $this->actingAs($this->admin, 'api')
            ->withoutExceptionHandling()
            ->getJson('/api/statistics/status-distribution?start_date=2024-01-01&end_date=2024-12-31');

        $response->assertStatus(200)
            ->assertJsonStructure([
                'data' => [
                    'completed',
                    'in_progress',
                    'pending',
                    'cancelled'
                ]
            ]);
    }

    #[Test]
    public function test_unauthorized_access_to_statistics()
    {
        // Rolsüz bir kullanıcı oluştur
        $user = User::factory()->create([
            'email' => 'user_' . Str::random(10) . '@example.com'
        ]);

        $response = $this->actingAs($user, 'api')
            ->getJson('/api/statistics/status-distribution');

        $response->assertStatus(403);
    }
}