<?php

namespace Tests\Feature;

use App\Models\ServiceRecord;
use App\Models\User;
use App\Notifications\ServiceStatusUpdated;
use App\Notifications\TechnicianAssigned;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Mail;
use PHPUnit\Framework\Attributes\Test;
use Tests\TestCase;

class ServiceRecordNotificationTest extends TestCase
{
    use RefreshDatabase;

    protected $admin;
    protected $serviceManager;
    protected $serviceStaff;
    protected $technician;
    protected $customer;
    protected $serviceRecord;

    protected function setUp(): void
    {
        parent::setUp();

        // Kullanıcı rollerini oluştur
        $this->admin = User::factory()->create(['role' => 'admin']);
        $this->serviceManager = User::factory()->create(['role' => 'service_manager']);
        $this->serviceStaff = User::factory()->create(['role' => 'service_staff']);
        $this->technician = User::factory()->create(['role' => 'technician']);
        $this->customer = User::factory()->create(['role' => 'customer']);

        // Test için bir servis kaydı oluştur
        $this->serviceRecord = ServiceRecord::factory()->create([
            'customer_email' => $this->customer->email,
            'status' => 'pending'
        ]);
    }

    #[Test]
    public function durum_degisikliginde_bildirim_gonderilir()
    {
        Mail::fake();

        $this->serviceRecord->changeStatus('in_progress', 'İşleme alındı', $this->serviceManager, true);

        Mail::assertSent(ServiceStatusUpdated::class, function ($mail) {
            return $mail->serviceRecord->id === $this->serviceRecord->id &&
                   $mail->serviceRecord->status === 'in_progress' &&
                   $mail->oldStatus === 'pending';
        });
    }

    #[Test]
    public function bildirim_devre_disiyken_durum_degisikliginde_bildirim_gonderilmez()
    {
        Mail::fake();

        $this->serviceRecord->changeStatus('in_progress', 'İşleme alındı', $this->serviceManager, false);

        Mail::assertNotSent(ServiceStatusUpdated::class);
    }

    #[Test]
    public function teknik_servis_personeli_atandiginda_bildirim_gonderilir()
    {
        Mail::fake();

        $this->serviceRecord->assignTechnician($this->technician, $this->serviceManager, null, true);

        Mail::assertSent(TechnicianAssigned::class, function ($mail) {
            return $mail->serviceRecord->id === $this->serviceRecord->id &&
                   $mail->technician->id === $this->technician->id &&
                   $mail->assignedBy->id === $this->serviceManager->id;
        });
    }

    #[Test]
    public function teknik_servis_personeli_atamasi_devre_disiyken_bildirim_gonderilmez()
    {
        Mail::fake();

        $this->serviceRecord->assignTechnician($this->technician, $this->serviceManager, null, false);

        Mail::assertNotSent(TechnicianAssigned::class);
    }

    #[Test]
    public function servis_tamamlandiginda_bildirim_gonderilir()
    {
        Mail::fake();

        $this->serviceRecord->changeStatus('completed', 'Servis tamamlandı', $this->serviceManager, true);

        Mail::assertSent(ServiceStatusUpdated::class, function ($mail) {
            return $mail->serviceRecord->id === $this->serviceRecord->id &&
                   $mail->serviceRecord->status === 'completed';
        });
    }

    #[Test]
    public function servis_iptal_edildiginde_bildirim_gonderilir()
    {
        Mail::fake();

        $this->serviceRecord->changeStatus('cancelled', 'Servis iptal edildi', $this->serviceManager, true);

        Mail::assertSent(ServiceStatusUpdated::class, function ($mail) {
            return $mail->serviceRecord->id === $this->serviceRecord->id &&
                   $mail->serviceRecord->status === 'cancelled';
        });
    }

    #[Test]
    public function parca_beklenirken_bildirim_gonderilir()
    {
        Mail::fake();

        $this->serviceRecord->changeStatus('waiting_for_parts', 'Parça bekleniyor', $this->serviceManager, true);

        Mail::assertSent(ServiceStatusUpdated::class, function ($mail) {
            return $mail->serviceRecord->id === $this->serviceRecord->id &&
                   $mail->serviceRecord->status === 'waiting_for_parts';
        });
    }

    #[Test]
    public function durum_degisikliginde_musteriye_bildirim_gonderilir()
    {
        Mail::fake();

        $this->serviceRecord->changeStatus('in_progress', 'İşleme alındı', $this->serviceManager, true);

        Mail::assertSent(ServiceStatusUpdated::class, function ($mail) {
            return $mail->serviceRecord->customer_email === $this->customer->email;
        });
    }

    #[Test]
    public function teknik_servis_atamasinda_musteriye_bildirim_gonderilir()
    {
        Mail::fake();

        $this->serviceRecord->assignTechnician($this->technician, $this->serviceManager, null, true);

        Mail::assertSent(TechnicianAssigned::class, function ($mail) {
            return $mail->serviceRecord->customer_email === $this->customer->email;
        });
    }
} 