<?php

namespace Tests\Feature;

use App\Models\ServiceRecord;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class ServiceRecordAuthorizationTest extends TestCase
{
    use RefreshDatabase;

    protected function setUp(): void
    {
        parent::setUp();
    }

    public function test_admin_can_view_all_service_records()
    {
        $admin = User::factory()->create([
            'role' => 'admin',
            'email' => 'admin1@example.com'
        ]);

        $this->actingAs($admin);

        $response = $this->get(route('service-records.index'));
        $response->assertStatus(200);
        $response->assertViewHas('serviceRecords');
    }

    public function test_service_manager_can_view_all_service_records()
    {
        $serviceManager = User::factory()->create([
            'role' => 'service_manager',
            'email' => 'manager1@example.com'
        ]);

        $this->actingAs($serviceManager);

        $response = $this->get(route('service-records.index'));
        $response->assertStatus(200);
        $response->assertViewHas('serviceRecords');
    }

    public function test_service_staff_can_view_all_service_records()
    {
        $serviceStaff = User::factory()->create([
            'role' => 'service_staff',
            'email' => 'staff1@example.com'
        ]);

        $this->actingAs($serviceStaff);

        $response = $this->get(route('service-records.index'));
        $response->assertStatus(200);
        $response->assertViewHas('serviceRecords');
    }

    public function test_customer_can_only_view_own_service_records()
    {
        $customer = User::factory()->create([
            'role' => 'customer',
            'email' => 'customer1@example.com'
        ]);

        $serviceRecord = ServiceRecord::factory()->create([
            'status' => ServiceRecord::STATUS_PENDING,
            'customer_email' => $customer->email
        ]);

        $this->actingAs($customer);

        // Kendi servis kaydını görüntüleyebilmeli
        $response = $this->get(route('service-records.show', $serviceRecord));
        $response->assertStatus(200);

        // Başka bir servis kaydı oluştur
        $otherServiceRecord = ServiceRecord::factory()->create([
            'customer_email' => 'other@example.com'
        ]);

        // Başka bir servis kaydını görüntüleyememeli
        $response = $this->get(route('service-records.show', $otherServiceRecord));
        $response->assertStatus(403);
    }

    public function test_admin_can_create_service_record()
    {
        $admin = User::factory()->create(['role' => 'admin']);

        $response = $this->actingAs($admin)
            ->post(route('service-records.store'), [
                'customer_name' => 'Test Customer',
                'customer_phone' => '5551234567',
                'customer_email' => 'test@example.com',
                'device_brand' => 'Test Brand',
                'device_model' => 'Test Model',
                'device_serial_number' => 'SN123456',
                'fault_description' => 'Test Description',
                'status' => ServiceRecord::STATUS_PENDING
            ]);

        $serviceRecord = ServiceRecord::latest()->first();
        $response->assertRedirect(route('service-records.show', $serviceRecord));
    }

    public function test_service_manager_can_create_service_record()
    {
        $serviceManager = User::factory()->create(['role' => 'service_manager']);

        $response = $this->actingAs($serviceManager)
            ->post(route('service-records.store'), [
                'customer_name' => 'Test Customer',
                'customer_phone' => '5551234567',
                'customer_email' => 'test@example.com',
                'device_brand' => 'Test Brand',
                'device_model' => 'Test Model',
                'device_serial_number' => 'SN123456',
                'fault_description' => 'Test Description',
                'status' => ServiceRecord::STATUS_PENDING
            ]);

        $serviceRecord = ServiceRecord::latest()->first();
        $response->assertRedirect(route('service-records.show', $serviceRecord));
    }

    public function test_service_staff_can_create_service_record()
    {
        $serviceStaff = User::factory()->create(['role' => 'service_staff']);

        $response = $this->actingAs($serviceStaff)
            ->post(route('service-records.store'), [
                'customer_name' => 'Test Customer',
                'customer_phone' => '5551234567',
                'customer_email' => 'test@example.com',
                'device_brand' => 'Test Brand',
                'device_model' => 'Test Model',
                'device_serial_number' => 'SN123456',
                'fault_description' => 'Test Description',
                'status' => ServiceRecord::STATUS_PENDING
            ]);

        $serviceRecord = ServiceRecord::latest()->first();
        $response->assertRedirect(route('service-records.show', $serviceRecord));
    }

    public function test_customer_cannot_create_service_record()
    {
        $customer = User::factory()->create([
            'role' => 'customer',
            'email' => 'customer2@example.com'
        ]);

        $this->actingAs($customer);

        $response = $this->get(route('service-records.create'));
        $response->assertStatus(403);

        $response = $this->post(route('service-records.store'), [
            'customer_name' => 'Test Customer',
            'customer_phone' => '5551234567',
            'device_model' => 'Test Device',
            'device_serial_number' => 'SN123456',
            'fault_description' => 'Test Description',
            'status' => ServiceRecord::STATUS_PENDING
        ]);

        $response->assertStatus(403);
    }

    public function test_admin_can_update_any_service_record()
    {
        $admin = User::factory()->create(['role' => 'admin']);
        $serviceRecord = ServiceRecord::factory()->create();

        $response = $this->actingAs($admin)
            ->put(route('service-records.update', $serviceRecord), [
                'customer_name' => 'Updated Customer',
                'customer_phone' => '5551234567',
                'customer_email' => 'updated@example.com',
                'device_brand' => 'Test Brand',
                'device_model' => 'Updated Model',
                'device_serial_number' => 'SN123456',
                'fault_description' => 'Updated Description',
                'status' => ServiceRecord::STATUS_IN_PROGRESS
            ]);

        $response->assertRedirect(route('service-records.show', $serviceRecord));
    }

    public function test_service_manager_can_update_any_service_record()
    {
        $serviceManager = User::factory()->create(['role' => 'service_manager']);
        $serviceRecord = ServiceRecord::factory()->create();

        $response = $this->actingAs($serviceManager)
            ->put(route('service-records.update', $serviceRecord), [
                'customer_name' => 'Updated Customer',
                'customer_phone' => '5551234567',
                'customer_email' => 'updated@example.com',
                'device_brand' => 'Test Brand',
                'device_model' => 'Updated Model',
                'device_serial_number' => 'SN123456',
                'fault_description' => 'Updated Description',
                'status' => ServiceRecord::STATUS_IN_PROGRESS
            ]);

        $response->assertRedirect(route('service-records.show', $serviceRecord));
    }

    public function test_service_staff_can_update_any_service_record()
    {
        $serviceStaff = User::factory()->create(['role' => 'service_staff']);
        $serviceRecord = ServiceRecord::factory()->create();

        $response = $this->actingAs($serviceStaff)
            ->put(route('service-records.update', $serviceRecord), [
                'customer_name' => 'Updated Customer',
                'customer_phone' => '5551234567',
                'customer_email' => 'updated@example.com',
                'device_brand' => 'Test Brand',
                'device_model' => 'Updated Model',
                'device_serial_number' => 'SN123456',
                'fault_description' => 'Updated Description',
                'status' => ServiceRecord::STATUS_IN_PROGRESS
            ]);

        $response->assertRedirect(route('service-records.show', $serviceRecord));
    }

    public function test_customer_cannot_update_service_record()
    {
        $customer = User::factory()->create([
            'role' => 'customer',
            'email' => 'customer6@example.com'
        ]);

        $serviceRecord = ServiceRecord::factory()->create([
            'status' => ServiceRecord::STATUS_PENDING,
            'customer_email' => $customer->email
        ]);

        $this->actingAs($customer);

        $response = $this->get(route('service-records.edit', $serviceRecord));
        $response->assertStatus(403);

        $response = $this->put(route('service-records.update', $serviceRecord), [
            'customer_name' => 'Updated Customer',
            'customer_phone' => '5551234567',
            'device_model' => 'Updated Device',
            'device_serial_number' => 'SN123456',
            'fault_description' => 'Updated Description',
            'status' => ServiceRecord::STATUS_IN_PROGRESS
        ]);

        $response->assertStatus(403);
    }

    public function test_admin_can_change_status()
    {
        $admin = User::factory()->create([
            'role' => 'admin',
            'email' => 'admin4@example.com'
        ]);

        $customer = User::factory()->create([
            'role' => 'customer',
            'email' => 'customer7@example.com'
        ]);

        $serviceRecord = ServiceRecord::factory()->create([
            'status' => ServiceRecord::STATUS_PENDING,
            'customer_email' => $customer->email
        ]);

        $this->actingAs($admin);

        $response = $this->post(route('service-records.change-status', $serviceRecord), [
            'new_status' => ServiceRecord::STATUS_IN_PROGRESS,
            'description' => 'Status changed by admin',
            'notify_customer' => true
        ]);

        $response->assertRedirect(route('service-records.show', $serviceRecord));
    }

    public function test_service_manager_can_change_status()
    {
        $serviceManager = User::factory()->create(['role' => 'service_manager']);
        $serviceRecord = ServiceRecord::factory()->create();

        $response = $this->actingAs($serviceManager)
            ->post(route('service-records.change-status', $serviceRecord), [
                'new_status' => 'in_progress',
                'description' => 'Test status change'
            ]);

        $response->assertRedirect(route('service-records.show', $serviceRecord));
        $this->assertEquals('in_progress', $serviceRecord->fresh()->status);
    }

    public function test_service_staff_can_change_status()
    {
        $serviceStaff = User::factory()->create([
            'role' => 'service_staff',
            'email' => 'staff4@example.com'
        ]);

        $customer = User::factory()->create([
            'role' => 'customer',
            'email' => 'customer9@example.com'
        ]);

        $serviceRecord = ServiceRecord::factory()->create([
            'status' => ServiceRecord::STATUS_PENDING,
            'customer_email' => $customer->email
        ]);

        $this->actingAs($serviceStaff);

        $response = $this->post(route('service-records.change-status', $serviceRecord), [
            'new_status' => ServiceRecord::STATUS_IN_PROGRESS,
            'description' => 'Status changed by staff',
            'notify_customer' => true
        ]);

        $response->assertRedirect(route('service-records.show', $serviceRecord));
    }

    public function test_customer_cannot_change_status()
    {
        $customer = User::factory()->create([
            'role' => 'customer',
            'email' => 'customer10@example.com'
        ]);

        $serviceRecord = ServiceRecord::factory()->create([
            'status' => ServiceRecord::STATUS_PENDING,
            'customer_email' => $customer->email
        ]);

        $this->actingAs($customer);

        $response = $this->post(route('service-records.change-status', $serviceRecord), [
            'new_status' => ServiceRecord::STATUS_IN_PROGRESS,
            'description' => 'Status changed by customer',
            'notify_customer' => true
        ]);

        $response->assertStatus(403);
    }

    public function test_admin_can_assign_technician()
    {
        $admin = User::factory()->create([
            'role' => 'admin',
            'email' => 'admin5@example.com'
        ]);

        $technician = User::factory()->create([
            'role' => 'technician',
            'email' => 'technician1@example.com'
        ]);

        $customer = User::factory()->create([
            'role' => 'customer',
            'email' => 'customer11@example.com'
        ]);

        $serviceRecord = ServiceRecord::factory()->create([
            'status' => ServiceRecord::STATUS_PENDING,
            'customer_email' => $customer->email
        ]);

        $this->actingAs($admin);

        $response = $this->post(route('service-records.assign-technician', $serviceRecord), [
            'technician_id' => $technician->id,
            'notify_customer' => true
        ]);

        $response->assertRedirect(route('service-records.show', $serviceRecord));
    }

    public function test_service_manager_can_assign_technician()
    {
        $serviceManager = User::factory()->create([
            'role' => 'service_manager',
            'email' => 'manager5@example.com'
        ]);

        $technician = User::factory()->create([
            'role' => 'technician',
            'email' => 'technician2@example.com'
        ]);

        $customer = User::factory()->create([
            'role' => 'customer',
            'email' => 'customer12@example.com'
        ]);

        $serviceRecord = ServiceRecord::factory()->create([
            'status' => ServiceRecord::STATUS_PENDING,
            'customer_email' => $customer->email
        ]);

        $this->actingAs($serviceManager);

        $response = $this->post(route('service-records.assign-technician', $serviceRecord), [
            'technician_id' => $technician->id,
            'notify_customer' => true
        ]);

        $response->assertRedirect(route('service-records.show', $serviceRecord));
    }

    public function test_service_staff_can_assign_technician()
    {
        $serviceStaff = User::factory()->create([
            'role' => 'service_staff',
            'email' => 'staff5@example.com'
        ]);

        $technician = User::factory()->create([
            'role' => 'technician',
            'email' => 'technician3@example.com'
        ]);

        $customer = User::factory()->create([
            'role' => 'customer',
            'email' => 'customer13@example.com'
        ]);

        $serviceRecord = ServiceRecord::factory()->create([
            'status' => ServiceRecord::STATUS_PENDING,
            'customer_email' => $customer->email
        ]);

        $this->actingAs($serviceStaff);

        $response = $this->post(route('service-records.assign-technician', $serviceRecord), [
            'technician_id' => $technician->id,
            'notify_customer' => true
        ]);

        $response->assertRedirect(route('service-records.show', $serviceRecord));
    }

    public function test_customer_cannot_assign_technician()
    {
        $customer = User::factory()->create([
            'role' => 'customer',
            'email' => 'customer14@example.com'
        ]);

        $technician = User::factory()->create([
            'role' => 'technician',
            'email' => 'technician4@example.com'
        ]);

        $serviceRecord = ServiceRecord::factory()->create([
            'status' => ServiceRecord::STATUS_PENDING,
            'customer_email' => $customer->email
        ]);

        $this->actingAs($customer);

        $response = $this->post(route('service-records.assign-technician', $serviceRecord), [
            'technician_id' => $technician->id,
            'notify_customer' => true
        ]);

        $response->assertStatus(403);
    }

    public function test_technician_can_view_assigned_records()
    {
        $technician = User::factory()->create([
            'role' => 'technician',
            'email' => 'technician5@example.com'
        ]);

        $assignedRecord = ServiceRecord::factory()->create([
            'technician_id' => $technician->id
        ]);

        $this->actingAs($technician);

        $response = $this->get(route('service-records.show', $assignedRecord));
        $response->assertStatus(200);
    }

    public function test_technician_cannot_view_unassigned_records()
    {
        $technician = User::factory()->create([
            'role' => 'technician',
            'email' => 'technician6@example.com'
        ]);

        $unassignedRecord = ServiceRecord::factory()->create([
            'technician_id' => null
        ]);

        $this->actingAs($technician);

        $response = $this->get(route('service-records.show', $unassignedRecord));
        $response->assertStatus(403);
    }
} 