<?php

namespace Tests\Feature;

use App\Models\ServiceRecord;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use PHPUnit\Framework\Attributes\Test;
use Tests\TestCase;

class ServiceRecordApiTest extends TestCase
{
    use RefreshDatabase;

    protected $admin;
    protected $serviceManager;
    protected $serviceStaff;
    protected $technician;
    protected $customer;
    protected $serviceRecord;

    protected function setUp(): void
    {
        parent::setUp();

        // Kullanıcı rollerini oluştur
        $this->admin = User::factory()->create(['role' => 'admin']);
        $this->serviceManager = User::factory()->create(['role' => 'service_manager']);
        $this->serviceStaff = User::factory()->create(['role' => 'service_staff']);
        $this->technician = User::factory()->create(['role' => 'technician']);
        $this->customer = User::factory()->create(['role' => 'customer']);

        // Test için bir servis kaydı oluştur
        $this->serviceRecord = ServiceRecord::factory()->create([
            'customer_email' => $this->customer->email,
            'status' => 'pending'
        ]);
    }

    #[Test]
    public function admin_servis_kayitlarini_listeleyebilir()
    {
        $this->actingAs($this->admin);

        $response = $this->getJson('/api/service-records');

        $response->assertStatus(200)
                ->assertJsonStructure([
                    'data' => [
                        '*' => [
                            'id',
                            'tracking_number',
                            'customer_name',
                            'customer_email',
                            'status',
                            'created_at',
                            'updated_at'
                        ]
                    ],
                    'meta' => [
                        'current_page',
                        'per_page',
                        'total'
                    ]
                ]);
    }

    #[Test]
    public function servis_yoneticisi_servis_kayitlarini_listeleyebilir()
    {
        $this->actingAs($this->serviceManager);

        $response = $this->getJson('/api/service-records');

        $response->assertStatus(200);
    }

    #[Test]
    public function servis_personeli_servis_kayitlarini_listeleyebilir()
    {
        $this->actingAs($this->serviceStaff);

        $response = $this->getJson('/api/service-records');

        $response->assertStatus(200);
    }

    #[Test]
    public function teknik_servis_personeli_sadece_kendisine_atanan_kayitlari_gorebilir()
    {
        $this->actingAs($this->technician);

        // Mevcut kayıtları temizle
        ServiceRecord::truncate();

        // Teknisyene atanmış kayıt
        $assignedRecord = ServiceRecord::factory()->create([
            'customer_email' => $this->customer->email,
            'technician_id' => $this->technician->id,
            'status' => 'in_progress'
        ]);

        $response = $this->getJson('/api/service-records');

        $response->assertStatus(200)
                ->assertJsonCount(1, 'data')
                ->assertJsonFragment(['id' => $assignedRecord->id]);
    }

    #[Test]
    public function musteri_sadece_kendi_kayitlarini_gorebilir()
    {
        $this->actingAs($this->customer);

        // Mevcut kayıtları temizle
        ServiceRecord::truncate();

        // Müşterinin kaydı
        $customerRecord = ServiceRecord::factory()->create([
            'customer_email' => $this->customer->email,
            'status' => 'pending'
        ]);

        // Başka bir müşterinin kaydı
        $otherCustomer = User::factory()->create(['role' => 'customer']);
        ServiceRecord::factory()->create([
            'customer_email' => $otherCustomer->email,
            'status' => 'pending'
        ]);

        $response = $this->getJson('/api/service-records');

        $response->assertStatus(200)
                ->assertJsonCount(1, 'data')
                ->assertJsonFragment(['id' => $customerRecord->id]);
    }

    #[Test]
    public function admin_yeni_servis_kaydi_olusturabilir()
    {
        $this->actingAs($this->admin);

        $response = $this->postJson('/api/service-records', [
            'customer_name' => 'Test Müşteri',
            'customer_email' => 'test@example.com',
            'customer_phone' => '5551234567',
            'customer_address' => 'Test Adres',
            'device_brand' => 'Test Marka',
            'device_model' => 'Test Model',
            'device_serial_number' => 'TEST123',
            'fault_description' => 'Test Arıza Açıklaması',
            'status' => 'pending'
        ]);

        $response->assertStatus(201)
                ->assertJsonStructure([
                    'data' => [
                        'id',
                        'tracking_number',
                        'customer_name',
                        'customer_email',
                        'status',
                        'created_at',
                        'updated_at'
                    ]
                ]);
    }

    #[Test]
    public function servis_kaydi_olustururken_gerekli_alanlar_dogrulanir()
    {
        $this->actingAs($this->admin);

        $response = $this->postJson('/api/service-records', []);

        $response->assertStatus(422)
                ->assertJsonValidationErrors([
                    'customer_name',
                    'customer_email',
                    'customer_phone',
                    'device_brand',
                    'device_model',
                    'device_serial_number',
                    'fault_description',
                    'status'
                ]);
    }

    #[Test]
    public function admin_servis_kaydini_guncelleyebilir()
    {
        $this->actingAs($this->admin);

        $response = $this->putJson("/api/service-records/{$this->serviceRecord->id}", [
            'customer_name' => $this->serviceRecord->customer_name,
            'customer_email' => $this->serviceRecord->customer_email,
            'customer_phone' => $this->serviceRecord->customer_phone,
            'customer_address' => $this->serviceRecord->customer_address,
            'device_brand' => $this->serviceRecord->device_brand,
            'device_model' => $this->serviceRecord->device_model,
            'device_serial_number' => $this->serviceRecord->device_serial_number,
            'status' => 'in_progress',
            'fault_description' => 'Güncellenmiş Arıza Açıklaması'
        ]);

        $response->assertStatus(200)
                ->assertJsonFragment([
                    'status' => 'in_progress',
                    'fault_description' => 'Güncellenmiş Arıza Açıklaması'
                ]);
    }

    #[Test]
    public function teknik_servis_personeli_kendisine_atanan_kaydi_guncelleyebilir()
    {
        $this->actingAs($this->technician);

        $assignedRecord = ServiceRecord::factory()->create([
            'customer_email' => $this->customer->email,
            'technician_id' => $this->technician->id,
            'status' => 'in_progress'
        ]);

        $response = $this->putJson("/api/service-records/{$assignedRecord->id}", [
            'customer_name' => $assignedRecord->customer_name,
            'customer_email' => $assignedRecord->customer_email,
            'customer_phone' => $assignedRecord->customer_phone,
            'customer_address' => $assignedRecord->customer_address,
            'device_brand' => $assignedRecord->device_brand,
            'device_model' => $assignedRecord->device_model,
            'device_serial_number' => $assignedRecord->device_serial_number,
            'status' => 'waiting_for_parts',
            'fault_description' => 'Parça bekleniyor'
        ]);

        $response->assertStatus(200);
    }

    #[Test]
    public function teknik_servis_personeli_kendisine_atanmayan_kaydi_guncelleyemez()
    {
        $this->actingAs($this->technician);

        $response = $this->putJson("/api/service-records/{$this->serviceRecord->id}", [
            'customer_name' => $this->serviceRecord->customer_name,
            'customer_email' => $this->serviceRecord->customer_email,
            'customer_phone' => $this->serviceRecord->customer_phone,
            'customer_address' => $this->serviceRecord->customer_address,
            'device_brand' => $this->serviceRecord->device_brand,
            'device_model' => $this->serviceRecord->device_model,
            'device_serial_number' => $this->serviceRecord->device_serial_number,
            'status' => 'in_progress',
            'fault_description' => 'Güncellenmiş Arıza Açıklaması'
        ]);

        $response->assertStatus(403);
    }

    #[Test]
    public function musteri_kendi_kaydini_guncelleyemez()
    {
        $this->actingAs($this->customer);

        $response = $this->putJson("/api/service-records/{$this->serviceRecord->id}", [
            'customer_name' => $this->serviceRecord->customer_name,
            'customer_email' => $this->serviceRecord->customer_email,
            'customer_phone' => $this->serviceRecord->customer_phone,
            'customer_address' => $this->serviceRecord->customer_address,
            'device_brand' => $this->serviceRecord->device_brand,
            'device_model' => $this->serviceRecord->device_model,
            'device_serial_number' => $this->serviceRecord->device_serial_number,
            'status' => 'in_progress',
            'fault_description' => 'Güncellenmiş Arıza Açıklaması'
        ]);

        $response->assertStatus(403);
    }

    #[Test]
    public function admin_servis_kaydini_silebilir()
    {
        $this->actingAs($this->admin);

        $response = $this->deleteJson("/api/service-records/{$this->serviceRecord->id}");

        $response->assertStatus(204);
        $this->assertSoftDeleted($this->serviceRecord);
    }

    #[Test]
    public function servis_yoneticisi_servis_kaydini_silebilir()
    {
        $this->actingAs($this->serviceManager);

        $response = $this->deleteJson("/api/service-records/{$this->serviceRecord->id}");

        $response->assertStatus(204);
    }

    #[Test]
    public function servis_personeli_servis_kaydini_silebilir()
    {
        $this->actingAs($this->serviceStaff);

        $response = $this->deleteJson("/api/service-records/{$this->serviceRecord->id}");

        $response->assertStatus(204);
    }

    #[Test]
    public function teknik_servis_personeli_servis_kaydini_silemez()
    {
        $this->actingAs($this->technician);

        $response = $this->deleteJson("/api/service-records/{$this->serviceRecord->id}");

        $response->assertStatus(403);
    }

    #[Test]
    public function musteri_servis_kaydini_silemez()
    {
        $this->actingAs($this->customer);

        $response = $this->deleteJson("/api/service-records/{$this->serviceRecord->id}");

        $response->assertStatus(403);
    }
} 