<?php

namespace Tests\Feature;

use App\Models\ServiceRecord;
use App\Models\ServiceStatusChange;
use App\Models\User;
use App\Notifications\ServiceStatusUpdated;
use App\Notifications\TechnicianAssigned;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Notification;
use Tests\TestCase;

class ServiceNotificationTest extends TestCase
{
    use RefreshDatabase;

    protected $admin;
    protected $technician;
    protected $customer;
    protected $serviceRecord;

    protected function setUp(): void
    {
        parent::setUp();

        // Bildirimleri takip et
        Notification::fake();

        // Admin kullanıcısı oluştur
        $this->admin = User::factory()->create([
            'role' => 'admin',
            'email' => 'admin' . uniqid() . '@example.com'
        ]);

        // Teknik servis personeli oluştur
        $this->technician = User::factory()->create([
            'role' => 'technician',
            'email' => 'technician' . uniqid() . '@example.com'
        ]);

        // Müşteri oluştur
        $this->customer = User::factory()->create([
            'role' => 'customer',
            'email' => 'customer' . uniqid() . '@example.com'
        ]);

        // Servis kaydı oluştur
        $this->serviceRecord = ServiceRecord::factory()->create([
            'status' => ServiceRecord::STATUS_PENDING,
            'customer_email' => $this->customer->email
        ]);
    }

    public function test_service_status_change_notification()
    {
        $this->actingAs($this->admin);

        // Durum değişikliği yap
        $this->serviceRecord->changeStatus(
            ServiceRecord::STATUS_IN_PROGRESS,
            'İşleme alındı',
            $this->admin
        );

        // Müşteriye bildirim gönderildiğini kontrol et
        Notification::assertSentTo(
            $this->customer,
            ServiceStatusUpdated::class,
            function ($notification) {
                return $notification->serviceRecord->id === $this->serviceRecord->id &&
                       $notification->oldStatus === ServiceRecord::STATUS_PENDING &&
                       $notification->newStatus === ServiceRecord::STATUS_IN_PROGRESS &&
                       $notification->description === 'İşleme alındı';
            }
        );
    }

    public function test_technician_assignment_notification()
    {
        $customer = User::factory()->create(['role' => 'customer']);
        $serviceRecord = ServiceRecord::factory()->create(['customer_email' => $customer->email]);
        $technician = User::factory()->create(['role' => 'technician']);
        $admin = User::factory()->create(['role' => 'admin']);

        $serviceRecord->assignTechnician($technician, $admin);

        Notification::assertSentTo(
            $customer,
            TechnicianAssigned::class,
            function ($notification) use ($serviceRecord, $technician, $admin) {
                return $notification->serviceRecord->id === $serviceRecord->id
                    && $notification->technician->id === $technician->id
                    && $notification->assignedBy->id === $admin->id;
            }
        );
    }

    public function test_notification_sent_to_technician()
    {
        $customer = User::factory()->create(['role' => 'customer']);
        $serviceRecord = ServiceRecord::factory()->create(['customer_email' => $customer->email]);
        $technician = User::factory()->create(['role' => 'technician']);
        $admin = User::factory()->create(['role' => 'admin']);

        $serviceRecord->assignTechnician($technician, $admin);

        Notification::assertSentTo(
            $technician,
            TechnicianAssigned::class,
            function ($notification) use ($serviceRecord, $technician, $admin) {
                return $notification->serviceRecord->id === $serviceRecord->id
                    && $notification->technician->id === $technician->id
                    && $notification->assignedBy->id === $admin->id;
            }
        );
    }

    public function test_notification_data_structure()
    {
        $this->actingAs($this->admin);

        // Durum değişikliği yap
        $this->serviceRecord->changeStatus(
            ServiceRecord::STATUS_IN_PROGRESS,
            'İşleme alındı',
            $this->admin
        );

        // Bildirim verilerinin doğru yapıda olduğunu kontrol et
        Notification::assertSentTo(
            $this->customer,
            ServiceStatusUpdated::class,
            function ($notification) {
                $data = $notification->toArray($this->customer);
                
                return isset($data['service_record_id']) &&
                       isset($data['tracking_number']) &&
                       isset($data['old_status']) &&
                       isset($data['new_status']) &&
                       isset($data['description']);
            }
        );
    }

    public function test_notification_not_sent_when_customer_not_found()
    {
        $serviceRecord = ServiceRecord::factory()->create(['customer_email' => 'nonexistent@example.com']);
        $technician = User::factory()->create(['role' => 'technician']);
        $admin = User::factory()->create(['role' => 'admin']);

        $serviceRecord->assignTechnician($technician, $admin);

        // Sadece teknisyene bildirim gönderildiğini kontrol et
        Notification::assertSentTimes(TechnicianAssigned::class, 1);
        
        // Teknisyene gönderilen bildirimi kontrol et
        Notification::assertSentTo(
            $technician,
            TechnicianAssigned::class,
            function ($notification) use ($serviceRecord, $technician, $admin) {
                return $notification->serviceRecord->id === $serviceRecord->id
                    && $notification->technician->id === $technician->id
                    && $notification->assignedBy->id === $admin->id;
            }
        );
    }
} 