<?php

namespace Tests\Feature;

use App\Models\User;
use App\Notifications\ServiceStatusUpdated;
use App\Notifications\TechnicianAssigned;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Notification;
use Tests\TestCase;

class NotificationTest extends TestCase
{
    use RefreshDatabase;

    protected $user;

    protected function setUp(): void
    {
        parent::setUp();
        $this->user = User::factory()->create();
        $this->actingAs($this->user);
    }

    public function test_user_can_view_notifications()
    {
        $response = $this->get(route('notifications.index'));
        $response->assertSuccessful();
    }

    public function test_user_can_mark_notification_as_read()
    {
        $notification = $this->user->notifications()->create([
            'id' => \Illuminate\Support\Str::uuid(),
            'type' => ServiceStatusUpdated::class,
            'data' => ['message' => 'Test notification']
        ]);

        $response = $this->post(route('notifications.read', $notification->id));
        $response->assertRedirect(route('notifications.index'));

        $this->assertNotNull($notification->fresh()->read_at);
    }

    public function test_user_can_mark_all_notifications_as_read()
    {
        $this->user->notifications()->create([
            'id' => \Illuminate\Support\Str::uuid(),
            'type' => ServiceStatusUpdated::class,
            'data' => ['message' => 'Test notification 1']
        ]);

        $this->user->notifications()->create([
            'id' => \Illuminate\Support\Str::uuid(),
            'type' => TechnicianAssigned::class,
            'data' => ['message' => 'Test notification 2']
        ]);

        $response = $this->post(route('notifications.read-all'));
        $response->assertRedirect(route('notifications.index'));

        $this->assertEquals(0, $this->user->unreadNotifications()->count());
    }

    public function test_user_can_delete_notification()
    {
        $notification = $this->user->notifications()->create([
            'id' => \Illuminate\Support\Str::uuid(),
            'type' => ServiceStatusUpdated::class,
            'data' => ['message' => 'Test notification']
        ]);

        $response = $this->delete(route('notifications.destroy', $notification->id));
        $response->assertRedirect(route('notifications.index'));

        $this->assertNull($notification->fresh());
    }

    public function test_user_can_delete_all_notifications()
    {
        $this->user->notifications()->create([
            'id' => \Illuminate\Support\Str::uuid(),
            'type' => ServiceStatusUpdated::class,
            'data' => ['message' => 'Test notification 1']
        ]);

        $this->user->notifications()->create([
            'id' => \Illuminate\Support\Str::uuid(),
            'type' => TechnicianAssigned::class,
            'data' => ['message' => 'Test notification 2']
        ]);

        $response = $this->delete(route('notifications.destroy-all'));
        $response->assertRedirect(route('notifications.index'));

        $this->assertEquals(0, $this->user->notifications()->count());
    }
} 