<?php

namespace Tests\Feature\Api;

use App\Models\ServiceRecord;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Foundation\Testing\WithFaker;
use Laravel\Sanctum\Sanctum;
use Tests\TestCase;
use Illuminate\Support\Facades\DB;

class ServiceRecordApiTest extends TestCase
{
    use RefreshDatabase, WithFaker;

    protected $admin;
    protected $serviceManager;
    protected $serviceStaff;
    protected $technician;
    protected $customer;
    protected $serviceRecord;

    protected function setUp(): void
    {
        parent::setUp();

        // Mevcut transaction'ı temizle
        while (DB::transactionLevel() > 0) {
            DB::rollBack();
        }

        // Admin kullanıcısı oluştur
        $this->admin = User::factory()->create([
            'role' => 'admin',
            'email' => 'admin' . uniqid() . '@example.com'
        ]);

        // Servis yöneticisi oluştur
        $this->serviceManager = User::factory()->create([
            'role' => 'service_manager',
            'email' => 'manager' . uniqid() . '@example.com'
        ]);

        // Servis personeli oluştur
        $this->serviceStaff = User::factory()->create([
            'role' => 'service_staff',
            'email' => 'staff' . uniqid() . '@example.com'
        ]);

        // Teknik servis personeli oluştur
        $this->technician = User::factory()->create([
            'role' => 'technician',
            'email' => 'technician' . uniqid() . '@example.com'
        ]);

        // Müşteri oluştur
        $this->customer = User::factory()->create([
            'role' => 'customer',
            'email' => 'customer' . uniqid() . '@example.com'
        ]);
    }

    protected function tearDown(): void
    {
        // Mevcut transaction'ı temizle
        while (DB::transactionLevel() > 0) {
            DB::rollBack();
        }

        parent::tearDown();
    }

    public function test_admin_can_list_service_records()
    {
        $serviceRecords = ServiceRecord::factory()->count(3)->create();

        $response = $this->actingAs($this->admin)
            ->getJson('/api/service-records');

        $response->assertStatus(200)
            ->assertJsonStructure([
                'success',
                'data' => [
                    'current_page',
                    'data' => [
                        '*' => [
                            'id',
                            'tracking_number',
                            'customer_name',
                            'customer_phone',
                            'customer_email',
                            'status',
                            'created_at',
                            'updated_at'
                        ]
                    ],
                    'first_page_url',
                    'from',
                    'last_page',
                    'last_page_url',
                    'next_page_url',
                    'path',
                    'per_page',
                    'prev_page_url',
                    'to',
                    'total'
                ]
            ]);
    }

    public function test_technician_can_list_service_records()
    {
        $serviceRecords = ServiceRecord::factory()->count(3)->create();

        $response = $this->actingAs($this->technician)
            ->getJson('/api/service-records');

        $response->assertStatus(200)
            ->assertJsonStructure([
                'success',
                'data' => [
                    'current_page',
                    'data' => [
                        '*' => [
                            'id',
                            'tracking_number',
                            'customer_name',
                            'customer_phone',
                            'customer_email',
                            'status',
                            'created_at',
                            'updated_at'
                        ]
                    ],
                    'first_page_url',
                    'from',
                    'last_page',
                    'last_page_url',
                    'next_page_url',
                    'path',
                    'per_page',
                    'prev_page_url',
                    'to',
                    'total'
                ]
            ]);
    }

    public function test_customer_can_only_see_own_records()
    {
        $ownRecord = ServiceRecord::factory()->create([
            'customer_email' => $this->customer->email
        ]);

        $otherRecord = ServiceRecord::factory()->create([
            'customer_email' => 'other@example.com'
        ]);

        $response = $this->actingAs($this->customer)
            ->getJson('/api/service-records');

        $response->assertStatus(200)
            ->assertJsonCount(1, 'data.data')
            ->assertJsonFragment(['id' => $ownRecord->id])
            ->assertJsonMissing(['id' => $otherRecord->id]);
    }

    public function test_admin_can_create_service_record()
    {
        $data = [
            'customer_name' => $this->faker->name,
            'customer_phone' => $this->faker->phoneNumber,
            'customer_email' => $this->faker->email,
            'customer_address' => $this->faker->address,
            'device_brand' => $this->faker->company,
            'device_model' => $this->faker->word,
            'device_serial_number' => $this->faker->uuid,
            'fault_description' => $this->faker->sentence,
            'status' => ServiceRecord::STATUS_PENDING
        ];

        $response = $this->actingAs($this->admin)
            ->postJson('/api/service-records', $data);

        $response->assertStatus(201)
            ->assertJson([
                'data' => $data
            ]);
    }

    public function test_admin_can_update_service_record()
    {
        $serviceRecord = ServiceRecord::factory()->create([
            'customer_email' => $this->customer->email
        ]);

        $data = [
            'customer_name' => $this->faker->name,
            'customer_phone' => $this->faker->phoneNumber,
            'customer_email' => $this->faker->email,
            'customer_address' => $this->faker->address,
            'device_brand' => $this->faker->company,
            'device_model' => $this->faker->word,
            'device_serial_number' => $this->faker->uuid,
            'fault_description' => $this->faker->sentence,
            'status' => ServiceRecord::STATUS_IN_PROGRESS
        ];

        $response = $this->actingAs($this->admin)
            ->putJson("/api/service-records/{$serviceRecord->id}", $data);

        $response->assertStatus(200)
            ->assertJson([
                'data' => $data
            ]);
    }

    public function test_admin_can_change_service_record_status()
    {
        $serviceRecord = ServiceRecord::factory()->create([
            'customer_email' => $this->customer->email
        ]);

        $data = [
            'new_status' => ServiceRecord::STATUS_IN_PROGRESS,
            'description' => $this->faker->sentence,
            'notify_customer' => true
        ];

        $response = $this->actingAs($this->admin)
            ->postJson("/api/service-records/{$serviceRecord->id}/change-status", $data);

        $response->assertStatus(200)
            ->assertJson([
                'data' => [
                    'status' => ServiceRecord::STATUS_IN_PROGRESS
                ]
            ]);
    }

    public function test_admin_can_assign_technician()
    {
        $serviceRecord = ServiceRecord::factory()->create([
            'customer_email' => $this->customer->email
        ]);

        $data = [
            'technician_id' => $this->technician->id,
            'notify_customer' => true
        ];

        $response = $this->actingAs($this->admin)
            ->postJson("/api/service-records/{$serviceRecord->id}/assign-technician", $data);

        $response->assertStatus(200)
            ->assertJson([
                'data' => [
                    'technician_id' => $this->technician->id
                ]
            ]);
    }

    public function test_cannot_assign_non_technician()
    {
        $serviceRecord = ServiceRecord::factory()->create([
            'customer_email' => $this->customer->email
        ]);

        $data = [
            'technician_id' => $this->customer->id,
            'notify_customer' => true
        ];

        $response = $this->actingAs($this->admin)
            ->postJson("/api/service-records/{$serviceRecord->id}/assign-technician", $data);

        $response->assertStatus(422)
            ->assertJson([
                'message' => 'Seçilen kullanıcı bir teknik servis personeli değil.'
            ]);
    }

    public function test_admin_can_view_service_record_timeline()
    {
        $serviceRecord = ServiceRecord::factory()->create([
            'customer_email' => $this->customer->email
        ]);

        $response = $this->actingAs($this->admin)
            ->getJson("/api/service-records/{$serviceRecord->id}/timeline");

        $response->assertStatus(200)
            ->assertJsonStructure([
                'success',
                'data' => [
                    '*' => [
                        'id',
                        'service_record_id',
                        'old_status',
                        'new_status',
                        'description',
                        'creator_id',
                        'created_at',
                        'updated_at',
                        'creator' => [
                            'id',
                            'name',
                            'email'
                        ]
                    ]
                ]
            ]);
    }

    public function test_can_search_and_filter_service_records()
    {
        // Önce tüm kayıtları temizle
        ServiceRecord::truncate();

        // Bekleyen kayıt
        $pendingRecord = ServiceRecord::factory()->create([
            'customer_name' => 'John Doe',
            'status' => ServiceRecord::STATUS_PENDING
        ]);

        // Tamamlanmış kayıt
        $completedRecord = ServiceRecord::factory()->create([
            'customer_name' => 'Jane Smith',
            'status' => ServiceRecord::STATUS_COMPLETED
        ]);

        // Teknisyene atanmış kayıt
        $assignedRecord = ServiceRecord::factory()->create([
            'customer_name' => 'Bob Wilson',
            'status' => ServiceRecord::STATUS_IN_PROGRESS,
            'technician_id' => $this->technician->id
        ]);

        // Arama testi
        $response = $this->actingAs($this->admin)
            ->getJson('/api/service-records?search=John');
        
        $response->assertStatus(200)
            ->assertJsonCount(1, 'data.data')
            ->assertJsonPath('data.data.0.id', $pendingRecord->id);

        // Durum filtresi testi
        $response = $this->actingAs($this->admin)
            ->getJson('/api/service-records?status=' . ServiceRecord::STATUS_COMPLETED);
        
        $response->assertStatus(200)
            ->assertJsonCount(1, 'data.data')
            ->assertJsonPath('data.data.0.id', $completedRecord->id);

        // Teknisyen filtresi testi
        $response = $this->actingAs($this->admin)
            ->getJson('/api/service-records?technician_id=' . $this->technician->id);
        
        $response->assertStatus(200)
            ->assertJsonCount(1, 'data.data')
            ->assertJsonPath('data.data.0.id', $assignedRecord->id);
    }

    public function test_can_sort_service_records()
    {
        // Önce tüm kayıtları temizle
        ServiceRecord::truncate();

        $oldRecord = ServiceRecord::factory()->create([
            'created_at' => now()->subDays(2)
        ]);

        $middleRecord = ServiceRecord::factory()->create([
            'created_at' => now()->subDay()
        ]);

        $newRecord = ServiceRecord::factory()->create([
            'created_at' => now()
        ]);

        // Tarihe göre sıralama (en yeni)
        $response = $this->actingAs($this->admin)
            ->getJson('/api/service-records?sort=created_at&direction=desc');

        $response->assertStatus(200)
            ->assertJsonCount(3, 'data.data')
            ->assertJsonPath('data.data.0.id', $newRecord->id)
            ->assertJsonPath('data.data.1.id', $middleRecord->id)
            ->assertJsonPath('data.data.2.id', $oldRecord->id);

        // Tarihe göre sıralama (en eski)
        $response = $this->actingAs($this->admin)
            ->getJson('/api/service-records?sort=created_at&direction=asc');

        $response->assertStatus(200)
            ->assertJsonCount(3, 'data.data')
            ->assertJsonPath('data.data.0.id', $oldRecord->id)
            ->assertJsonPath('data.data.1.id', $middleRecord->id)
            ->assertJsonPath('data.data.2.id', $newRecord->id);
    }
} 