<?php

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Route;
use App\Http\Controllers\Api\AuthController;
use App\Http\Controllers\Api\ServiceRecordController;
use App\Http\Controllers\Api\ServiceItemController;
use App\Http\Controllers\Api\ServiceStatusController;
use App\Http\Controllers\Api\TechnicianController;
use App\Http\Controllers\CustomerController;
use App\Http\Controllers\DashboardController;
use App\Http\Controllers\ServiceRecordStatisticsController;
use App\Http\Controllers\Api\Customer\CustomerDashboardController;
use App\Http\Controllers\Api\Customer\CustomerServiceRecordController;
use App\Http\Controllers\Api\Customer\CustomerProfileController;
use App\Http\Controllers\Api\Customer\CustomerNotificationController;

/*
|--------------------------------------------------------------------------
| API Routes
|--------------------------------------------------------------------------
|
| Here is where you can register API routes for your application. These
| routes are loaded by the RouteServiceProvider and all of them will
| be assigned to the "api" middleware group. Make something great!
|
*/

Route::middleware('auth:sanctum')->get('/user', function (Request $request) {
    return $request->user();
});

// Public routes
Route::post('/register', [AuthController::class, 'register']);
Route::post('/login', [AuthController::class, 'login']);

// Protected routes
Route::middleware('auth:sanctum')->group(function () {
    // Auth routes
    Route::post('/logout', [AuthController::class, 'logout']);
    Route::get('/me', [AuthController::class, 'me']);

    // Customer routes
    Route::apiResource('customers', CustomerController::class);
    
    // Service Records Routes
    Route::prefix('service-records')->group(function () {
        Route::get('/reports', [ServiceRecordController::class, 'report']);
        Route::get('/', [ServiceRecordController::class, 'index']);
        Route::post('/', [ServiceRecordController::class, 'store']);
        Route::get('/{serviceRecord}', [ServiceRecordController::class, 'show']);
        Route::put('/{serviceRecord}', [ServiceRecordController::class, 'update']);
        Route::delete('/{serviceRecord}', [ServiceRecordController::class, 'destroy']);
        Route::post('/{serviceRecord}/change-status', [ServiceRecordController::class, 'changeStatus']);
        Route::post('/{serviceRecord}/assign-technician', [ServiceRecordController::class, 'assignTechnician']);
        Route::get('/{serviceRecord}/timeline', [ServiceRecordController::class, 'timeline']);

        // İstatistik rotaları
        Route::middleware('role:admin,service_manager,service_staff')->group(function () {
            Route::get('/statistics/status-distribution', [ServiceRecordController::class, 'statusDistributionStatistics']);
            Route::get('/statistics/technician-performance', [ServiceRecordController::class, 'technicianPerformanceStatistics']);
            Route::get('/statistics/completion-time', [ServiceRecordController::class, 'completionTimeStatistics']);
            Route::get('/statistics/monthly', [ServiceRecordController::class, 'monthlyStatistics']);
            Route::get('/statistics/customers', [ServiceRecordController::class, 'customerStatistics']);
        });

        // Export rotaları
        Route::get('/export/excel', [ServiceRecordController::class, 'exportToExcel']);
        Route::get('/export/pdf', [ServiceRecordController::class, 'exportToPdf']);
    });

    Route::middleware(['auth:sanctum', 'role:admin,technician'])->group(function () {
        // Sadece admin ve teknik servis personeli için erişilebilir rotalar
        Route::post('/service-records', [ServiceRecordController::class, 'store']);
        Route::put('/service-records/{serviceRecord}', [ServiceRecordController::class, 'update']);
        Route::post('/service-records/{serviceRecord}/change-status', [ServiceRecordController::class, 'changeStatus']);
        Route::post('/service-records/{serviceRecord}/assign-technician', [ServiceRecordController::class, 'assignTechnician']);
    });

    // Service Items Routes
    Route::middleware('role:admin,technician')->group(function () {
        Route::apiResource('service-records.items', ServiceItemController::class)->shallow();
        Route::post('service-items/{item}/add-specification', [ServiceItemController::class, 'addSpecification']);
        Route::post('service-items/{item}/add-accessory', [ServiceItemController::class, 'addAccessory']);
        Route::delete('service-items/{item}/remove-accessory', [ServiceItemController::class, 'removeAccessory']);
    });

    // Service Status Routes
    Route::middleware('role:admin,technician')->group(function () {
        Route::apiResource('service-records.statuses', ServiceStatusController::class)->shallow();
        Route::get('service-records/{serviceRecord}/latest-status', [ServiceStatusController::class, 'latest']);
    });

    // Technician Routes
    Route::middleware('role:admin')->group(function () {
        Route::apiResource('technicians', TechnicianController::class);
        Route::get('technicians/{technician}/assigned-services', [TechnicianController::class, 'assignedServices']);
        Route::get('technicians/{technician}/workload', [TechnicianController::class, 'workload']);
        Route::post('service-records/{serviceRecord}/assign', [TechnicianController::class, 'assignService']);
        Route::delete('service-records/{serviceRecord}/unassign', [TechnicianController::class, 'unassignService']);
    });

    // Dashboard routes
    Route::get('/dashboard/stats', [DashboardController::class, 'stats']);
    Route::get('/dashboard/recent-services', [DashboardController::class, 'recentServices']);

    // Service Record Delete Route
    Route::middleware('role:admin,service_manager,service_staff')->group(function () {
        Route::delete('/service-records/{serviceRecord}', [ServiceRecordController::class, 'destroy']);
    });

    Route::middleware(['auth:sanctum', 'role:admin,service_manager,service_staff'])->group(function () {
        Route::get('/service-records/statistics/status-distribution', [ServiceRecordController::class, 'statusDistributionStatistics']);
        Route::get('/service-records/statistics/technician-performance', [ServiceRecordController::class, 'technicianPerformanceStatistics']);
        Route::get('/service-records/statistics/completion-time', [ServiceRecordController::class, 'completionTimeStatistics']);
        Route::get('/service-records/statistics/monthly', [ServiceRecordController::class, 'monthlyStatistics']);
        Route::get('/service-records/statistics/customers', [ServiceRecordController::class, 'customerStatistics']);
    });

    // İstatistik rotaları
    Route::middleware(['auth:sanctum'])->group(function () {
        Route::prefix('service-records/statistics')->group(function () {
            Route::get('/status-distribution', [ServiceRecordController::class, 'statusDistributionStatistics']);
            Route::get('/technician-performance', [ServiceRecordController::class, 'technicianPerformanceStatistics']);
            Route::get('/completion-time', [ServiceRecordController::class, 'completionTimeStatistics']);
            Route::get('/monthly', [ServiceRecordController::class, 'monthlyStatistics']);
            Route::get('/customers', [ServiceRecordController::class, 'customerStatistics']);
        });
    });
});

Route::middleware('auth:api')->get('/user', function (Request $request) {
    return $request->user();
});

Route::prefix('statistics')->group(function () {
    Route::get('status-distribution', [ServiceRecordStatisticsController::class, 'statusDistribution']);
    Route::get('technician-performance', [ServiceRecordStatisticsController::class, 'technicianPerformance']);
    Route::get('completion-time', [ServiceRecordStatisticsController::class, 'completionTime']);
    Route::get('monthly', [ServiceRecordStatisticsController::class, 'monthly']);
    Route::get('customers', [ServiceRecordStatisticsController::class, 'customers']);
});

// Müşteri Portalı Route'ları
Route::middleware(['auth:sanctum', 'role:customer'])->prefix('customer')->name('customer.')->group(function () {
    // Dashboard
    Route::get('/dashboard', [CustomerDashboardController::class, 'index'])->name('dashboard');
    
    // Servis Kayıtları
    Route::get('/service-records', [CustomerServiceRecordController::class, 'index'])->name('service-records.index');
    Route::get('/service-records/{serviceRecord}', [CustomerServiceRecordController::class, 'show'])->name('service-records.show');
    
    // Profil Yönetimi
    Route::get('/profile', [CustomerProfileController::class, 'edit'])->name('profile.edit');
    Route::put('/profile', [CustomerProfileController::class, 'update'])->name('profile.update');
    Route::put('/profile/password', [CustomerProfileController::class, 'updatePassword'])->name('profile.password');
    
    // Bildirim Tercihleri
    Route::get('/notifications', [CustomerNotificationController::class, 'index'])->name('notifications.index');
    Route::put('/notifications', [CustomerNotificationController::class, 'update'])->name('notifications.update');
});