<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Technician extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'user_id',
        'specialties',
        'is_available',
        'max_concurrent_tasks',
        'working_hours'
    ];

    protected $casts = [
        'specialties' => 'json',
        'working_hours' => 'json',
        'is_available' => 'boolean'
    ];

    // Relationships
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function serviceRecords()
    {
        return $this->hasMany(ServiceRecord::class, 'assigned_technician_id', 'user_id');
    }

    // Scopes
    public function scopeAvailable($query)
    {
        return $query->where('is_available', true);
    }

    public function scopeSpecialty($query, $specialty)
    {
        return $query->whereJsonContains('specialties', $specialty);
    }

    // Methods
    public function addSpecialty($specialty)
    {
        $specialties = $this->specialties ?? [];
        if (!in_array($specialty, $specialties)) {
            $specialties[] = $specialty;
            $this->specialties = $specialties;
            $this->save();
        }
        return $this;
    }

    public function removeSpecialty($specialty)
    {
        $specialties = $this->specialties ?? [];
        $this->specialties = array_values(array_diff($specialties, [$specialty]));
        $this->save();
        return $this;
    }

    public function setWorkingHours(array $hours)
    {
        $this->working_hours = $hours;
        $this->save();
        return $this;
    }

    public function getCurrentWorkload()
    {
        return $this->serviceRecords()
                    ->whereNotIn('status', ['completed', 'delivered', 'cancelled'])
                    ->count();
    }

    public function canTakeNewTask()
    {
        return $this->is_available && $this->getCurrentWorkload() < $this->max_concurrent_tasks;
    }
}
