<?php

namespace App\Models;

use App\Events\ServiceStatusChanged;
use App\Notifications\TechnicianAssigned;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Notifications\Notifiable;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Facades\DB;
use App\Notifications\ServiceUnassigned;
use App\Notifications\ServiceStatusUpdated;
use Illuminate\Support\Facades\Notification;
use Illuminate\Support\Facades\Mail;
use App\Notifications\ServiceRecordStatusChanged;

class ServiceRecord extends Model
{
    use HasFactory, SoftDeletes, Notifiable;

    const STATUS_PENDING = 'pending';
    const STATUS_IN_PROGRESS = 'in_progress';
    const STATUS_WAITING_FOR_PARTS = 'waiting_for_parts';
    const STATUS_COMPLETED = 'completed';
    const STATUS_CANCELLED = 'cancelled';

    const STATUSES = [
        self::STATUS_PENDING => 'Beklemede',
        self::STATUS_IN_PROGRESS => 'İşlemde',
        self::STATUS_WAITING_FOR_PARTS => 'Parça Bekliyor',
        self::STATUS_COMPLETED => 'Tamamlandı',
        self::STATUS_CANCELLED => 'İptal Edildi'
    ];

    protected $fillable = [
        'tracking_number',
        'customer_id',
        'customer_name',
        'customer_phone',
        'customer_email',
        'customer_address',
        'device_brand',
        'device_model',
        'device_serial_number',
        'fault_description',
        'status',
        'technician_id',
        'notes'
    ];

    protected $casts = [
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
        'deleted_at' => 'datetime'
    ];

    // Relationships
    public function technician(): BelongsTo
    {
        return $this->belongsTo(User::class, 'technician_id');
    }

    public function customer(): BelongsTo
    {
        return $this->belongsTo(User::class, 'customer_id');
    }

    public function statuses(): HasMany
    {
        return $this->hasMany(ServiceStatusChange::class)
            ->with('changedBy')
            ->orderBy('changed_at', 'desc');
    }

    // Scopes
    public function scopeSearch($query, ?string $search)
    {
        if (!$search) {
            return $query;
        }

        return $query->where(function($q) use ($search) {
            $q->where('tracking_number', 'like', "%{$search}%")
              ->orWhere('customer_name', 'like', "%{$search}%")
              ->orWhere('customer_phone', 'like', "%{$search}%")
              ->orWhere('customer_email', 'like', "%{$search}%");
        });
    }

    public function scopeFilterByStatus($query, ?string $status)
    {
        if (!$status) {
            return $query;
        }

        return $query->where('status', $status);
    }

    public function scopeFilterByTechnician($query, ?int $technicianId)
    {
        if (!$technicianId) {
            return $query;
        }

        return $query->where('technician_id', $technicianId);
    }

    public function scopeOrderByField($query, string $field, string $direction = 'asc')
    {
        $allowedFields = ['id', 'created_at', 'updated_at', 'status', 'tracking_number', 'customer_name'];
        if (!in_array($field, $allowedFields)) {
            $field = 'created_at';
        }

        return $query->orderBy($field, $direction);
    }

    // Methods
    /**
     * Change the status of the service record.
     *
     * @param string $newStatus
     * @param string|null $description
     * @param User|null $changedBy
     * @param bool $notifyCustomer
     * @return bool
     */
    public function changeStatus(string $newStatus, ?string $description = null, ?User $changedBy = null, bool $notifyCustomer = true): bool
    {
        if ($this->status === $newStatus) {
            return false;
        }

        $oldStatus = $this->status;
        $this->status = $newStatus;
        $this->save();

        // Durum değişikliğini kaydet
        $this->statuses()->create([
            'old_status' => $oldStatus,
            'new_status' => $newStatus,
            'changed_by_id' => $changedBy?->id ?? auth()->id(),
            'description' => $description
        ]);

        // Müşteriye bildirim gönder
        if ($notifyCustomer && $this->customer_email) {
            $notification = new ServiceStatusUpdated(
                $this,
                $oldStatus,
                $newStatus,
                $description ?? $newStatus,
                $changedBy
            );
            Mail::to($this->customer_email)->send($notification);
        }

        return true;
    }

    /**
     * Get the technician assignments for this service record.
     */
    public function technicianAssignments()
    {
        return $this->statuses()
            ->whereNotNull('technician_id')
            ->with('changedBy')
            ->orderBy('created_at', 'desc')
            ->get();
    }

    /**
     * Assign a technician to this service record.
     *
     * @param User $technician The technician to assign
     * @param User $assignedBy The user making the assignment (defaults to current user)
     * @param string|null $description Optional notes about the assignment
     * @param bool $notifyCustomer Whether to notify the customer
     * @return void
     * @throws \InvalidArgumentException If the user is not a technician or is deleted
     */
    public function assignTechnician(User $technician, User $assignedBy, ?string $description = null, bool $notifyCustomer = true)
    {
        if (!$technician->hasRole('technician')) {
            throw new \InvalidArgumentException('Seçilen kullanıcı bir teknik servis personeli değil.');
        }

        $this->technician_id = $technician->id;
        $this->save();

        // Durum değişikliğini kaydet
        $this->statuses()->create([
            'old_status' => $this->getOriginal('status'),
            'new_status' => $this->status,
            'description' => $description ?? 'Teknisyen atandı: ' . $technician->name,
            'changed_by_id' => $assignedBy->id,
            'changed_at' => now()
        ]);

        // Müşteriye bildirim gönder
        if ($notifyCustomer && $this->customer_email) {
            $notification = new TechnicianAssigned($this, $technician, $assignedBy);
            Mail::to($this->customer_email)->send($notification);
        }

        return $this;
    }

    /**
     * Unassign the current technician from this service record.
     *
     * @param string|null $notes Optional notes about the unassignment
     * @param User|null $unassignedBy The user removing the assignment (defaults to current user)
     * @return bool
     */
    public function unassignTechnician(?string $notes = null, ?User $unassignedBy = null): bool
    {
        if (!$this->technician_id) {
            return false;
        }

        $oldTechnician = $this->technician;
        $this->technician_id = null;
        $this->save();

        // Teknisyen atamasını kaldır
        $this->statuses()->create([
            'old_status' => $this->status,
            'new_status' => $this->status,
            'changed_by_id' => $unassignedBy?->id ?? auth()->id(),
            'description' => 'Teknisyen ataması kaldırıldı: ' . $oldTechnician->name
        ]);

        // Müşteriye bildirim gönder
        if ($this->customer_email) {
            $notification = new ServiceUnassigned($this, $oldTechnician);
            Mail::to($this->customer_email)->send($notification);
        }

        return true;
    }

    public function isAssigned(): bool
    {
        return $this->technician_id !== null;
    }

    public function isCompleted(): bool
    {
        return $this->status === self::STATUS_COMPLETED;
    }

    public function isCancelled(): bool
    {
        return $this->status === self::STATUS_CANCELLED;
    }

    public function isInProgress(): bool
    {
        return $this->status === self::STATUS_IN_PROGRESS;
    }

    public function isPending(): bool
    {
        return $this->status === self::STATUS_PENDING;
    }

    public function isWaitingForParts(): bool
    {
        return $this->status === self::STATUS_WAITING_FOR_PARTS;
    }

    protected static function booted()
    {
        static::creating(function ($serviceRecord) {
            $serviceRecord->tracking_number = $serviceRecord->generateTrackingNumber();
        });

        static::deleting(function ($serviceRecord) {
            if ($serviceRecord->isForceDeleting()) {
                $serviceRecord->statuses()->delete();
            }
        });
    }

    protected function generateTrackingNumber(): string
    {
        $prefix = 'SR';
        $year = now()->format('y');
        $month = now()->format('m');
        
        // Get the last service record number for this month
        $lastRecord = static::withTrashed()
            ->whereYear('created_at', now()->year)
            ->whereMonth('created_at', now()->month)
            ->orderByDesc('id')
            ->first();
        
        $sequence = $lastRecord ? (int)substr($lastRecord->tracking_number, -4) + 1 : 1;
        
        // Ensure the sequence is unique
        do {
            $trackingNumber = sprintf('%s%s%s%04d', $prefix, $year, $month, $sequence);
            $exists = static::withTrashed()->where('tracking_number', $trackingNumber)->exists();
            if ($exists) {
                $sequence++;
            }
        } while ($exists);
        
        return $trackingNumber;
    }

    public function getStatusTextAttribute(): string
    {
        return self::STATUSES[$this->status] ?? 'Bilinmiyor';
    }

    public function getStatusBadgeClassAttribute(): string
    {
        return match ($this->status) {
            self::STATUS_PENDING => 'bg-yellow-100 text-yellow-800',
            self::STATUS_IN_PROGRESS => 'bg-blue-100 text-blue-800',
            self::STATUS_WAITING_FOR_PARTS => 'bg-orange-100 text-orange-800',
            self::STATUS_COMPLETED => 'bg-green-100 text-green-800',
            self::STATUS_CANCELLED => 'bg-red-100 text-red-800',
            default => 'bg-gray-100 text-gray-800',
        };
    }

    /**
     * Get the status history for this service record
     */
    public function statusHistory()
    {
        return $this->statuses()
            ->with('changedBy')
            ->orderBy('changed_at', 'desc');
    }
}
