<?php

namespace App\Http\Controllers\Technician;

use App\Http\Controllers\Controller;
use App\Models\PartRequest;
use App\Models\ServiceRecord;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class TechnicianPartController extends Controller
{
    public function index(Request $request)
    {
        $technician = Auth::user();

        $query = PartRequest::whereHas('serviceRecord', function($q) use ($technician) {
                $q->where('technician_id', $technician->id);
            })
            ->with(['serviceRecord.customer'])
            ->orderBy('created_at', 'desc');

        // Filtreleme
        if ($request->has('status') && $request->status !== 'all') {
            $query->where('status', $request->status);
        }

        // Arama
        if ($request->has('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('part_name', 'like', "%{$search}%")
                    ->orWhere('part_code', 'like', "%{$search}%")
                    ->orWhereHas('serviceRecord', function($q) use ($search) {
                        $q->where('tracking_number', 'like', "%{$search}%")
                            ->orWhereHas('customer', function($q) use ($search) {
                                $q->where('name', 'like', "%{$search}%");
                            });
                    });
            });
        }

        $partRequests = $query->paginate(10);

        return view('technician.parts.index', compact('partRequests'));
    }

    public function request(Request $request, ServiceRecord $serviceRecord)
    {
        $this->authorize('update', $serviceRecord);

        $request->validate([
            'part_name' => 'required|string|max:255',
            'part_code' => 'required|string|max:255',
            'quantity' => 'required|integer|min:1',
            'description' => 'nullable|string|max:1000'
        ]);

        DB::transaction(function() use ($request, $serviceRecord) {
            $partRequest = PartRequest::create([
                'service_record_id' => $serviceRecord->id,
                'part_name' => $request->part_name,
                'part_code' => $request->part_code,
                'quantity' => $request->quantity,
                'description' => $request->description,
                'status' => 'pending'
            ]);

            // Servis kaydının durumunu güncelle
            $serviceRecord->update(['status' => 'waiting_for_parts']);

            // Durum değişikliği notu ekle
            $serviceRecord->notes()->create([
                'content' => "Parça talebi oluşturuldu: {$partRequest->part_name} (Kod: {$partRequest->part_code}, Adet: {$partRequest->quantity})",
                'type' => 'status_change'
            ]);
        });

        return back()->with('success', 'Parça talebi başarıyla oluşturuldu.');
    }

    public function updateStatus(Request $request, PartRequest $partRequest)
    {
        $this->authorize('update', $partRequest->serviceRecord);

        $request->validate([
            'status' => 'required|in:pending,approved,rejected,received',
            'note' => 'required|string|max:1000'
        ]);

        DB::transaction(function() use ($request, $partRequest) {
            $oldStatus = $partRequest->status;
            $partRequest->update(['status' => $request->status]);

            // Durum değişikliği notu ekle
            $partRequest->serviceRecord->notes()->create([
                'content' => "Parça talebi durumu güncellendi: {$partRequest->part_name} (Eski: {$oldStatus}, Yeni: {$request->status}) - {$request->note}",
                'type' => 'status_change'
            ]);

            // Eğer parça alındıysa ve başka bekleyen parça talebi yoksa servis durumunu güncelle
            if ($request->status === 'received') {
                $hasPendingParts = PartRequest::where('service_record_id', $partRequest->service_record_id)
                    ->where('status', '!=', 'received')
                    ->exists();

                if (!$hasPendingParts) {
                    $partRequest->serviceRecord->update(['status' => 'in_progress']);
                }
            }
        });

        return back()->with('success', 'Parça talebi durumu güncellendi.');
    }
} 