<?php

namespace App\Http\Controllers;

use App\Models\ServiceRecord;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Gate;
use Illuminate\Validation\Rule;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class ServiceRecordController extends Controller
{
    use AuthorizesRequests;

    public function index(Request $request)
    {
        $this->authorize('viewAny', ServiceRecord::class);

        $query = ServiceRecord::query();

        // Arama
        if ($request->has('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('tracking_number', 'like', "%{$search}%")
                  ->orWhere('customer_name', 'like', "%{$search}%")
                  ->orWhere('customer_phone', 'like', "%{$search}%")
                  ->orWhere('device_model', 'like', "%{$search}%")
                  ->orWhere('device_serial_number', 'like', "%{$search}%");
            });
        }

        // Filtreleme
        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        if ($request->has('technician_id')) {
            $query->where('technician_id', $request->technician_id);
        }

        if ($request->has('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }

        if ($request->has('date_to')) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        // Sıralama
        $sortField = $request->get('sort', 'created_at');
        $sortDirection = $request->get('direction', 'desc');
        $query->orderBy($sortField, $sortDirection);

        $serviceRecords = $query->paginate(10)->withQueryString();

        $technicians = User::technicians()->get();
        $statuses = ServiceRecord::STATUSES;

        return view('service-records.index', compact('serviceRecords', 'technicians', 'statuses'));
    }

    public function create()
    {
        $this->authorize('create', ServiceRecord::class);

        $technicians = User::technicians()->get();
        $statuses = ServiceRecord::STATUSES;

        return view('service-records.create', compact('technicians', 'statuses'));
    }

    public function store(Request $request)
    {
        $this->authorize('create', ServiceRecord::class);

        $validated = $request->validate([
            'customer_name' => ['required', 'string', 'max:255'],
            'customer_phone' => ['required', 'string', 'max:20'],
            'customer_email' => ['nullable', 'email', 'max:255'],
            'customer_address' => ['nullable', 'string', 'max:1000'],
            'device_brand' => ['required', 'string', 'max:255'],
            'device_model' => ['required', 'string', 'max:255'],
            'device_serial_number' => ['required', 'string', 'max:255'],
            'fault_description' => ['required', 'string', 'max:1000'],
            'status' => ['required', 'string', Rule::in(array_keys(ServiceRecord::STATUSES))],
            'technician_id' => ['nullable', 'exists:users,id'],
            'notes' => ['nullable', 'string', 'max:1000'],
        ]);

        // Teknisyen ataması kontrolü
        if ($request->has('technician_id')) {
            $technician = User::findOrFail($request->technician_id);
            if (!$technician->isTechnician()) {
                return back()->withErrors(['technician_id' => __('Seçilen kullanıcı teknik servis personeli değil.')]);
            }
        }

        $serviceRecord = ServiceRecord::create($validated);

        // İlk durum değişikliğini kaydet
        $serviceRecord->statusChanges()->create([
            'old_status' => 'created',
            'new_status' => $validated['status'],
            'changed_by_id' => auth()->id(),
            'description' => __('Servis kaydı oluşturuldu.')
        ]);

        return redirect()
            ->route('service-records.show', $serviceRecord)
            ->with('success', __('Servis kaydı başarıyla oluşturuldu.'));
    }

    public function show(ServiceRecord $serviceRecord)
    {
        $this->authorize('view', $serviceRecord);

        $serviceRecord->load(['technician', 'statusChanges.changedBy']);
        $technicians = User::technicians()->get();
        $statuses = ServiceRecord::STATUSES;

        return view('service-records.show', compact('serviceRecord', 'technicians', 'statuses'));
    }

    public function edit(ServiceRecord $serviceRecord)
    {
        $this->authorize('update', $serviceRecord);

        $technicians = User::technicians()->get();
        $statuses = ServiceRecord::STATUSES;

        return view('service-records.edit', compact('serviceRecord', 'technicians', 'statuses'));
    }

    public function update(Request $request, ServiceRecord $serviceRecord)
    {
        $this->authorize('update', $serviceRecord);

        $validated = $request->validate([
            'customer_name' => ['required', 'string', 'max:255'],
            'customer_phone' => ['required', 'string', 'max:20'],
            'customer_email' => ['nullable', 'email', 'max:255'],
            'customer_address' => ['nullable', 'string', 'max:1000'],
            'device_brand' => ['required', 'string', 'max:255'],
            'device_model' => ['required', 'string', 'max:255'],
            'device_serial_number' => ['required', 'string', 'max:255'],
            'fault_description' => ['required', 'string', 'max:1000'],
            'status' => ['required', 'string', Rule::in(array_keys(ServiceRecord::STATUSES))],
            'technician_id' => ['nullable', 'exists:users,id'],
            'notes' => ['nullable', 'string', 'max:1000'],
        ]);

        // Teknisyen ataması kontrolü
        if ($request->has('technician_id') && $request->technician_id != $serviceRecord->technician_id) {
            $technician = User::findOrFail($request->technician_id);
            if (!$technician->isTechnician()) {
                return back()->withErrors(['technician_id' => __('Seçilen kullanıcı teknik servis personeli değil.')]);
            }
        }

        // Durum değişikliği kontrolü
        if ($request->status != $serviceRecord->status) {
            $serviceRecord->statusChanges()->create([
                'old_status' => $serviceRecord->status,
                'new_status' => $request->status,
                'changed_by' => auth()->id(),
                'description' => __('Servis kaydı güncellendi.')
            ]);
        }

        $serviceRecord->update($validated);

        return redirect()
            ->route('service-records.show', $serviceRecord)
            ->with('success', __('Servis kaydı başarıyla güncellendi.'));
    }

    public function changeStatus(Request $request, ServiceRecord $serviceRecord)
    {
        $this->authorize('update', $serviceRecord);

        $validated = $request->validate([
            'new_status' => ['required', 'string', Rule::in(array_keys(ServiceRecord::STATUSES))],
            'description' => ['nullable', 'string', 'max:1000'],
            'notify_customer' => ['boolean']
        ]);

        $serviceRecord->changeStatus(
            $validated['new_status'],
            $validated['description'] ?? null,
            auth()->user(),
            $validated['notify_customer'] ?? true
        );

        return redirect()
            ->route('service-records.show', $serviceRecord)
            ->with('success', __('Servis durumu başarıyla güncellendi.'));
    }

    public function assignTechnician(ServiceRecord $serviceRecord, Request $request)
    {
        $this->authorize('update', $serviceRecord);

        $validated = $request->validate([
            'technician_id' => ['required', 'exists:users,id'],
            'notify_customer' => ['boolean'],
            'notes' => ['nullable', 'string', 'max:500']
        ]);

        $technician = User::findOrFail($validated['technician_id']);
        $notifyCustomer = $validated['notify_customer'] ?? true;

        $serviceRecord->assignTechnician(
            $technician,
            auth()->user(),
            $validated['notes'] ?? null,
            $notifyCustomer
        );

        return redirect()->route('service-records.show', $serviceRecord)
            ->with('success', __('Servis kaydına teknisyen atandı.'));
    }

    public function statusHistory(ServiceRecord $serviceRecord)
    {
        $this->authorize('view', $serviceRecord);

        $statusHistory = $serviceRecord->statusHistory()->orderBy('created_at', 'desc')->get();

        return view('service-records.status-history', compact('serviceRecord', 'statusHistory'));
    }

    public function destroy(ServiceRecord $serviceRecord)
    {
        $this->authorize('delete', $serviceRecord);

        $serviceRecord->delete();

        return redirect()->route('service-records.index')
            ->with('success', __('Servis kaydı başarıyla silindi.'));
    }

    public function technicianReport(Request $request)
    {
        $startDate = Carbon::parse($request->input('start_date', now()->subMonth()->startOfMonth()));
        $endDate = Carbon::parse($request->input('end_date', now()->endOfMonth()));

        $technicianReport = ServiceRecord::whereBetween('created_at', [$startDate, $endDate])
            ->select('technician_id', DB::raw('count(*) as total'))
            ->with('technician')
            ->groupBy('technician_id')
            ->get();

        return view('service-records.reports.technician', compact('technicianReport', 'startDate', 'endDate'));
    }

    public function customerReport(Request $request)
    {
        $startDate = Carbon::parse($request->input('start_date', now()->subMonth()->startOfMonth()));
        $endDate = Carbon::parse($request->input('end_date', now()->endOfMonth()));

        $customerReport = ServiceRecord::whereBetween('created_at', [$startDate, $endDate])
            ->select('customer_id', DB::raw('count(*) as total'))
            ->with('customer')
            ->groupBy('customer_id')
            ->get();

        return view('service-records.reports.customer', compact('customerReport', 'startDate', 'endDate'));
    }
} 