<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\ServiceRecord;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class TechnicianController extends Controller
{
    public function index(Request $request)
    {
        $query = User::where('role', 'technician');

        if ($request->has('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%");
            });
        }

        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        return response()->json([
            'success' => true,
            'data' => $query->paginate($request->input('per_page', 15))
        ]);
    }

    public function show(User $technician)
    {
        if ($technician->role !== 'technician') {
            return response()->json([
                'success' => false,
                'message' => 'Bu kullanıcı bir teknisyen değil.'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $technician->load(['assignedServices'])
        ]);
    }

    public function assignedServices(Request $request, User $technician)
    {
        if ($technician->role !== 'technician') {
            return response()->json([
                'success' => false,
                'message' => 'Bu kullanıcı bir teknisyen değil.'
            ], 404);
        }

        $query = $technician->assignedServices()
            ->with(['customer', 'items', 'statuses']);

        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        if ($request->has('priority')) {
            $query->where('priority', $request->priority);
        }

        if ($request->has('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }

        if ($request->has('date_to')) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        $query->orderBy('created_at', 'desc');

        return response()->json([
            'success' => true,
            'data' => $query->paginate($request->input('per_page', 15))
        ]);
    }

    public function assignService(Request $request, ServiceRecord $serviceRecord)
    {
        $validator = Validator::make($request->all(), [
            'technician_id' => 'required|exists:users,id'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $technician = User::find($request->technician_id);
        
        if ($technician->role !== 'technician') {
            return response()->json([
                'success' => false,
                'message' => 'Bu kullanıcı bir teknisyen değil.'
            ], 422);
        }

        $serviceRecord->assigned_technician_id = $technician->id;
        $serviceRecord->save();

        $serviceRecord->updateStatus(
            'assigned',
            'Teknisyen atandı: ' . $technician->name,
            auth()->id(),
            true
        );

        return response()->json([
            'success' => true,
            'data' => $serviceRecord->fresh(['technician']),
            'message' => 'Teknisyen başarıyla atandı.'
        ]);
    }

    public function unassignService(ServiceRecord $serviceRecord)
    {
        $serviceRecord->assigned_technician_id = null;
        $serviceRecord->save();

        $serviceRecord->updateStatus(
            'unassigned',
            'Teknisyen ataması kaldırıldı.',
            auth()->id(),
            true
        );

        return response()->json([
            'success' => true,
            'data' => $serviceRecord->fresh(),
            'message' => 'Teknisyen ataması başarıyla kaldırıldı.'
        ]);
    }

    public function workload(User $technician)
    {
        if ($technician->role !== 'technician') {
            return response()->json([
                'success' => false,
                'message' => 'Bu kullanıcı bir teknisyen değil.'
            ], 404);
        }

        $workload = [
            'total_services' => $technician->assignedServices()->count(),
            'active_services' => $technician->assignedServices()->where('status', '!=', 'completed')->count(),
            'completed_services' => $technician->assignedServices()->where('status', 'completed')->count(),
            'services_by_status' => $technician->assignedServices()
                ->selectRaw('status, count(*) as count')
                ->groupBy('status')
                ->get(),
            'services_by_priority' => $technician->assignedServices()
                ->selectRaw('priority, count(*) as count')
                ->groupBy('priority')
                ->get()
        ];

        return response()->json([
            'success' => true,
            'data' => $workload
        ]);
    }
} 