<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\ServiceItem;
use App\Models\ServiceRecord;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class ServiceItemController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request, ServiceRecord $serviceRecord)
    {
        $query = $serviceRecord->items();

        if ($request->has('type')) {
            $query->where('type', $request->type);
        }

        if ($request->has('brand')) {
            $query->where('brand', $request->brand);
        }

        return response()->json([
            'success' => true,
            'data' => $query->get()
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request, ServiceRecord $serviceRecord)
    {
        $validator = Validator::make($request->all(), [
            'type' => 'required|string',
            'brand' => 'required|string',
            'model' => 'required|string',
            'serial_number' => 'nullable|string',
            'description' => 'nullable|string',
            'specifications' => 'nullable|array',
            'condition_on_receipt' => 'nullable|string',
            'accessories' => 'nullable|array'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $item = $serviceRecord->items()->create($request->all());

        return response()->json([
            'success' => true,
            'data' => $item,
            'message' => 'Servis ürünü başarıyla eklendi.'
        ], 201);
    }

    /**
     * Display the specified resource.
     */
    public function show(ServiceItem $item)
    {
        return response()->json([
            'success' => true,
            'data' => $item->load('serviceRecord')
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, ServiceItem $item)
    {
        $validator = Validator::make($request->all(), [
            'type' => 'sometimes|required|string',
            'brand' => 'sometimes|required|string',
            'model' => 'sometimes|required|string',
            'serial_number' => 'nullable|string',
            'description' => 'nullable|string',
            'specifications' => 'nullable|array',
            'condition_on_receipt' => 'nullable|string',
            'accessories' => 'nullable|array'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $item->update($request->all());

        return response()->json([
            'success' => true,
            'data' => $item,
            'message' => 'Servis ürünü başarıyla güncellendi.'
        ]);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(ServiceItem $item)
    {
        $item->delete();

        return response()->json([
            'success' => true,
            'message' => 'Servis ürünü başarıyla silindi.'
        ]);
    }

    public function addSpecification(Request $request, ServiceItem $item)
    {
        $validator = Validator::make($request->all(), [
            'key' => 'required|string',
            'value' => 'required'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $item->addSpecification($request->key, $request->value);

        return response()->json([
            'success' => true,
            'data' => $item->fresh(),
            'message' => 'Özellik başarıyla eklendi.'
        ]);
    }

    public function addAccessory(Request $request, ServiceItem $item)
    {
        $validator = Validator::make($request->all(), [
            'accessory' => 'required|string'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $item->addAccessory($request->accessory);

        return response()->json([
            'success' => true,
            'data' => $item->fresh(),
            'message' => 'Aksesuar başarıyla eklendi.'
        ]);
    }

    public function removeAccessory(Request $request, ServiceItem $item)
    {
        $validator = Validator::make($request->all(), [
            'accessory' => 'required|string'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $item->removeAccessory($request->accessory);

        return response()->json([
            'success' => true,
            'data' => $item->fresh(),
            'message' => 'Aksesuar başarıyla kaldırıldı.'
        ]);
    }
}
